/*
 * Decompiled with CFR 0.152.
 */
package org.zalando.nakadi.repository.kafka;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import kafka.admin.AdminUtils;
import kafka.common.TopicExistsException;
import kafka.utils.ZkUtils;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.producer.BufferExhaustedException;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.SerializationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import org.zalando.nakadi.domain.BatchItem;
import org.zalando.nakadi.domain.Cursor;
import org.zalando.nakadi.domain.CursorError;
import org.zalando.nakadi.domain.EventPublishingStatus;
import org.zalando.nakadi.domain.EventPublishingStep;
import org.zalando.nakadi.domain.EventType;
import org.zalando.nakadi.domain.EventTypeOptions;
import org.zalando.nakadi.domain.EventTypeStatistics;
import org.zalando.nakadi.domain.SubscriptionBase;
import org.zalando.nakadi.domain.Topic;
import org.zalando.nakadi.exceptions.DuplicatedEventTypeNameException;
import org.zalando.nakadi.exceptions.EventPublishingException;
import org.zalando.nakadi.exceptions.InvalidCursorException;
import org.zalando.nakadi.exceptions.NakadiException;
import org.zalando.nakadi.exceptions.ServiceUnavailableException;
import org.zalando.nakadi.exceptions.TopicCreationException;
import org.zalando.nakadi.exceptions.TopicDeletionException;
import org.zalando.nakadi.repository.EventConsumer;
import org.zalando.nakadi.repository.TopicRepository;
import org.zalando.nakadi.repository.kafka.KafkaCursor;
import org.zalando.nakadi.repository.kafka.KafkaFactory;
import org.zalando.nakadi.repository.kafka.KafkaPartitionsCalculator;
import org.zalando.nakadi.repository.kafka.KafkaRepositorySettings;
import org.zalando.nakadi.repository.zookeeper.ZooKeeperHolder;

@Component
@Profile(value={"!test"})
public class KafkaTopicRepository
implements TopicRepository {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaTopicRepository.class);
    private final ZooKeeperHolder zkFactory;
    private final Producer<String, String> kafkaProducer;
    private final KafkaFactory kafkaFactory;
    private final KafkaRepositorySettings settings;
    private final KafkaPartitionsCalculator partitionsCalculator;

    @Autowired
    public KafkaTopicRepository(ZooKeeperHolder zkFactory, KafkaFactory kafkaFactory, KafkaRepositorySettings settings, KafkaPartitionsCalculator partitionsCalculator) {
        this.zkFactory = zkFactory;
        this.partitionsCalculator = partitionsCalculator;
        this.kafkaProducer = kafkaFactory.getProducer();
        this.kafkaFactory = kafkaFactory;
        this.settings = settings;
    }

    @Override
    public List<Topic> listTopics() throws ServiceUnavailableException {
        try {
            return ((List)this.zkFactory.get().getChildren().forPath("/brokers/topics")).stream().map(Topic::new).collect(Collectors.toList());
        }
        catch (Exception e) {
            throw new ServiceUnavailableException("Failed to list topics", e);
        }
    }

    @Override
    public void createTopic(EventType eventType) throws TopicCreationException, DuplicatedEventTypeNameException {
        this.createTopic(eventType.getTopic(), this.calculateKafkaPartitionCount(eventType.getDefaultStatistic()), this.settings.getDefaultTopicReplicaFactor(), this.getTopicRetentionMs(eventType), this.settings.getDefaultTopicRotationMs());
    }

    private long getTopicRetentionMs(EventType eventType) {
        return Optional.ofNullable(eventType.getOptions()).map(EventTypeOptions::getRetentionTime).orElse(this.settings.getDefaultTopicRetentionMs());
    }

    private void createTopic(String topic, int partitionsNum, int replicaFactor, long retentionMs, long rotationMs) throws TopicCreationException, DuplicatedEventTypeNameException {
        try {
            this.doWithZkUtils(zkUtils -> {
                Properties topicConfig = new Properties();
                topicConfig.setProperty("retention.ms", Long.toString(retentionMs));
                topicConfig.setProperty("segment.ms", Long.toString(rotationMs));
                AdminUtils.createTopic((ZkUtils)zkUtils, (String)topic, (int)partitionsNum, (int)replicaFactor, (Properties)topicConfig);
            });
        }
        catch (TopicExistsException e) {
            throw new DuplicatedEventTypeNameException("EventType with name " + topic + " already exists (or wasn't completely removed yet)");
        }
        catch (Exception e) {
            throw new TopicCreationException("Unable to create topic " + topic, e);
        }
    }

    @Override
    public void deleteTopic(String topic) throws TopicDeletionException {
        try {
            this.doWithZkUtils(zkUtils -> AdminUtils.deleteTopic((ZkUtils)zkUtils, (String)topic));
        }
        catch (Exception e) {
            throw new TopicDeletionException("Unable to delete topic " + topic, e);
        }
    }

    @Override
    public boolean topicExists(String topic) throws ServiceUnavailableException {
        return this.listTopics().stream().map(Topic::getName).anyMatch(t -> t.equals(topic));
    }

    @Override
    public boolean partitionExists(String topic, String partition) throws NakadiException {
        return this.listPartitionNames(topic).stream().anyMatch(partition::equals);
    }

    @Override
    public void syncPostBatch(String topicId, List<BatchItem> batch) throws EventPublishingException {
        CountDownLatch done = new CountDownLatch(batch.size());
        for (BatchItem item : batch) {
            ProducerRecord record = new ProducerRecord(topicId, Integer.valueOf(KafkaCursor.toKafkaPartition(item.getPartition())), (Object)item.getPartition(), (Object)item.getEvent().toString());
            item.setStep(EventPublishingStep.PUBLISHING);
            try {
                this.kafkaProducer.send(record, this.kafkaSendCallback(item, done));
            }
            catch (BufferExhaustedException | InterruptException | SerializationException e) {
                item.updateStatusAndDetail(EventPublishingStatus.FAILED, "internal error");
                throw new EventPublishingException("Error publishing message to kafka", (Exception)e);
            }
        }
        try {
            boolean isSuccessful = done.await(this.settings.getKafkaSendTimeoutMs(), TimeUnit.MILLISECONDS);
            if (!isSuccessful) {
                this.failBatch(batch, "timed out");
                throw new EventPublishingException("Timeout publishing events");
            }
        }
        catch (InterruptedException e) {
            this.failBatch(batch, "internal error");
            throw new EventPublishingException("Error publishing message to kafka", e);
        }
    }

    private void failBatch(List<BatchItem> batch, String reason) {
        for (BatchItem item : batch) {
            item.updateStatusAndDetail(EventPublishingStatus.FAILED, reason);
        }
    }

    private Callback kafkaSendCallback(BatchItem item, CountDownLatch done) {
        return (metadata, exception) -> {
            if (exception == null) {
                item.updateStatusAndDetail(EventPublishingStatus.SUBMITTED, "");
            } else {
                LOG.error("Failed to publish event", (Throwable)exception);
                item.updateStatusAndDetail(EventPublishingStatus.FAILED, "internal error");
            }
            done.countDown();
        };
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public List<org.zalando.nakadi.domain.TopicPartition> listPartitions(String topicId) throws ServiceUnavailableException {
        try (Consumer<String, String> consumer = this.kafkaFactory.getConsumer();){
            List<TopicPartition> kafkaTPs = consumer.partitionsFor(topicId).stream().map(p -> new TopicPartition(topicId, p.partition())).collect(Collectors.toList());
            consumer.assign(kafkaTPs);
            TopicPartition[] tpArray = kafkaTPs.toArray(new TopicPartition[kafkaTPs.size()]);
            consumer.seekToBeginning(tpArray);
            Map<Integer, Long> earliestOffsets = this.getPositions(consumer, kafkaTPs);
            consumer.seekToEnd(tpArray);
            Map<Integer, Long> latestOffsets = this.getPositions(consumer, kafkaTPs);
            List<org.zalando.nakadi.domain.TopicPartition> list = kafkaTPs.stream().map(tp -> {
                int partition = tp.partition();
                org.zalando.nakadi.domain.TopicPartition topicPartition = new org.zalando.nakadi.domain.TopicPartition(topicId, KafkaCursor.toNakadiPartition(partition));
                Long latestOffset = (Long)latestOffsets.get(partition);
                topicPartition.setNewestAvailableOffset(this.transformNewestOffset(latestOffset));
                topicPartition.setOldestAvailableOffset(KafkaCursor.toNakadiOffset((Long)earliestOffsets.get(partition)));
                return topicPartition;
            }).collect(Collectors.toList());
            return list;
        }
        catch (Exception e) {
            throw new ServiceUnavailableException("Error occurred when fetching partitions offsets", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public Map<String, Long> materializePositions(String topicId, SubscriptionBase.InitialPosition position) throws ServiceUnavailableException {
        try (Consumer<String, String> consumer = this.kafkaFactory.getConsumer();){
            TopicPartition[] kafkaTPs = (TopicPartition[])consumer.partitionsFor(topicId).stream().map(p -> new TopicPartition(topicId, p.partition())).toArray(TopicPartition[]::new);
            consumer.assign(Arrays.asList(kafkaTPs));
            if (position == SubscriptionBase.InitialPosition.BEGIN) {
                consumer.seekToBeginning(kafkaTPs);
            } else {
                if (position != SubscriptionBase.InitialPosition.END) throw new IllegalArgumentException("Bad offset specification " + (Object)((Object)position) + " for topic " + topicId);
                consumer.seekToEnd(kafkaTPs);
            }
            Map<String, Long> map = Stream.of(kafkaTPs).collect(Collectors.toMap(tp -> String.valueOf(tp.partition()), arg_0 -> consumer.position(arg_0)));
            return map;
        }
        catch (Exception e) {
            throw new ServiceUnavailableException("Error occurred when fetching partitions offsets", e);
        }
    }

    @Override
    public List<String> listPartitionNames(String topicId) {
        return Collections.unmodifiableList(this.kafkaFactory.getProducer().partitionsFor(topicId).stream().map(partitionInfo -> KafkaCursor.toNakadiPartition(partitionInfo.partition())).collect(Collectors.toList()));
    }

    private String transformNewestOffset(Long newestOffset) {
        return newestOffset == 0L ? "BEGIN" : KafkaCursor.toNakadiOffset(newestOffset - 1L);
    }

    private Map<Integer, Long> getPositions(Consumer<String, String> consumer, List<TopicPartition> kafkaTPs) {
        return kafkaTPs.stream().collect(Collectors.toMap(TopicPartition::partition, arg_0 -> consumer.position(arg_0)));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public org.zalando.nakadi.domain.TopicPartition getPartition(String topicId, String partition) throws ServiceUnavailableException {
        try (Consumer<String, String> consumer = this.kafkaFactory.getConsumer();){
            TopicPartition tp = new TopicPartition(topicId, KafkaCursor.toKafkaPartition(partition));
            consumer.assign((List)ImmutableList.of((Object)tp));
            org.zalando.nakadi.domain.TopicPartition topicPartition = new org.zalando.nakadi.domain.TopicPartition(topicId, partition);
            consumer.seekToBeginning(new TopicPartition[]{tp});
            topicPartition.setOldestAvailableOffset(KafkaCursor.toNakadiOffset(consumer.position(tp)));
            consumer.seekToEnd(new TopicPartition[]{tp});
            Long latestOffset = consumer.position(tp);
            topicPartition.setNewestAvailableOffset(this.transformNewestOffset(latestOffset));
            org.zalando.nakadi.domain.TopicPartition topicPartition2 = topicPartition;
            return topicPartition2;
        }
        catch (Exception e) {
            throw new ServiceUnavailableException("Error occurred when fetching partition offsets", e);
        }
    }

    public Consumer<String, String> createKafkaConsumer() {
        return this.kafkaFactory.getConsumer();
    }

    @Override
    public EventConsumer createEventConsumer(String topic, List<Cursor> cursors) throws ServiceUnavailableException, InvalidCursorException {
        this.validateCursors(topic, cursors);
        ArrayList kafkaCursors = Lists.newArrayListWithCapacity((int)cursors.size());
        for (Cursor cursor : cursors) {
            long kafkaOffset;
            String offset = cursor.getOffset();
            String partition = cursor.getPartition();
            if ("BEGIN".equals(offset)) {
                org.zalando.nakadi.domain.TopicPartition tp = this.getPartition(topic, partition);
                kafkaOffset = KafkaCursor.toKafkaOffset(tp.getOldestAvailableOffset());
            } else {
                kafkaOffset = KafkaCursor.toKafkaOffset(offset) + 1L;
            }
            KafkaCursor kafkaCursor = KafkaCursor.kafkaCursor(KafkaCursor.toKafkaPartition(partition), kafkaOffset);
            kafkaCursors.add(kafkaCursor);
        }
        return this.kafkaFactory.createNakadiConsumer(topic, kafkaCursors, this.settings.getKafkaPollTimeoutMs());
    }

    @Override
    public int compareOffsets(String firstOffset, String secondOffset) {
        try {
            long first = KafkaCursor.toKafkaOffset(firstOffset);
            long second = KafkaCursor.toKafkaOffset(secondOffset);
            return Long.compare(first, second);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Incorrect offset format, should be long", e);
        }
    }

    private void validateCursors(String topic, List<Cursor> cursors) throws ServiceUnavailableException, InvalidCursorException {
        List<org.zalando.nakadi.domain.TopicPartition> partitions = this.listPartitions(topic);
        for (Cursor cursor : cursors) {
            this.validateCursorForNulls(cursor);
            org.zalando.nakadi.domain.TopicPartition topicPartition = partitions.stream().filter(tp -> tp.getPartitionId().equals(cursor.getPartition())).findFirst().orElseThrow(() -> new InvalidCursorException(CursorError.PARTITION_NOT_FOUND, cursor));
            if ("BEGIN".equals(cursor.getOffset())) continue;
            if ("BEGIN".equals(topicPartition.getNewestAvailableOffset())) {
                throw new InvalidCursorException(CursorError.EMPTY_PARTITION, cursor);
            }
            long newestOffset = KafkaCursor.toKafkaOffset(topicPartition.getNewestAvailableOffset());
            long oldestOffset = KafkaCursor.toKafkaOffset(topicPartition.getOldestAvailableOffset());
            try {
                long offset = KafkaCursor.fromNakadiCursor(cursor).getOffset();
                if (offset >= oldestOffset - 1L && offset <= newestOffset) continue;
                throw new InvalidCursorException(CursorError.UNAVAILABLE, cursor);
            }
            catch (NumberFormatException e) {
                throw new InvalidCursorException(CursorError.INVALID_FORMAT, cursor);
            }
        }
    }

    @Override
    public void validateCommitCursors(String topic, List<Cursor> cursors) throws InvalidCursorException {
        List<String> partitions = this.listPartitionNames(topic);
        for (Cursor cursor : cursors) {
            this.validateCursorForNulls(cursor);
            if (!partitions.contains(cursor.getPartition())) {
                throw new InvalidCursorException(CursorError.PARTITION_NOT_FOUND, cursor);
            }
            try {
                KafkaCursor.fromNakadiCursor(cursor);
            }
            catch (NumberFormatException e) {
                throw new InvalidCursorException(CursorError.INVALID_FORMAT, cursor);
            }
        }
    }

    private void validateCursorForNulls(Cursor cursor) throws InvalidCursorException {
        if (cursor.getPartition() == null) {
            throw new InvalidCursorException(CursorError.NULL_PARTITION, cursor);
        }
        if (cursor.getOffset() == null) {
            throw new InvalidCursorException(CursorError.NULL_OFFSET, cursor);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doWithZkUtils(ZkUtilsAction action) throws Exception {
        try (ZkUtils zkUtils = null;){
            String connectionString = this.zkFactory.get().getZookeeperClient().getCurrentConnectionString();
            zkUtils = ZkUtils.apply((String)connectionString, (int)this.settings.getZkSessionTimeoutMs(), (int)this.settings.getZkConnectionTimeoutMs(), (boolean)false);
            action.execute(zkUtils);
        }
    }

    @VisibleForTesting
    Integer calculateKafkaPartitionCount(EventTypeStatistics stat) {
        if (null == stat) {
            return this.settings.getDefaultTopicPartitionCount();
        }
        int maxPartitionsDueParallelism = Math.max(stat.getReadParallelism(), stat.getWriteParallelism());
        if (maxPartitionsDueParallelism >= this.settings.getMaxTopicPartitionCount()) {
            return this.settings.getMaxTopicPartitionCount();
        }
        return Math.min(this.settings.getMaxTopicPartitionCount(), Math.max(maxPartitionsDueParallelism, this.calculatePartitionsAccordingLoad(stat.getMessagesPerMinute(), stat.getMessageSize())));
    }

    private int calculatePartitionsAccordingLoad(int messagesPerMinute, int avgEventSizeBytes) {
        float throughoutputMbPerSec = (float)messagesPerMinute * (float)avgEventSizeBytes / 6.291456E7f;
        return this.partitionsCalculator.getBestPartitionsCount(avgEventSizeBytes, throughoutputMbPerSec);
    }

    @FunctionalInterface
    private static interface ZkUtilsAction {
        public void execute(ZkUtils var1) throws Exception;
    }
}

