/*
 * Decompiled with CFR 0.152.
 */
package org.zalando.nakadi.service.subscription.state;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.LoggerFactory;
import org.zalando.nakadi.service.EventStream;
import org.zalando.nakadi.service.subscription.model.Partition;
import org.zalando.nakadi.service.subscription.state.CleanupState;
import org.zalando.nakadi.service.subscription.state.ClosingState;
import org.zalando.nakadi.service.subscription.state.PartitionData;
import org.zalando.nakadi.service.subscription.state.State;
import org.zalando.nakadi.service.subscription.zk.ZKSubscription;

class StreamingState
extends State {
    private ZKSubscription topologyChangeSubscription;
    private Consumer<String, String> kafkaConsumer;
    private final Map<Partition.PartitionKey, PartitionData> offsets = new HashMap<Partition.PartitionKey, PartitionData>();
    private final Map<Partition.PartitionKey, Long> releasingPartitions = new HashMap<Partition.PartitionKey, Long>();
    private boolean pollPaused = false;
    private long lastCommitMillis;
    private long committedEvents = 0L;
    private long sentEvents = 0L;

    StreamingState() {
    }

    @Override
    public void onEnter() {
        this.kafkaConsumer = this.getKafka().createKafkaConsumer();
        this.topologyChangeSubscription = this.getZk().subscribeForTopologyChanges(() -> this.addTask(this::topologyChanged));
        this.reactOnTopologyChange();
        this.addTask(this::pollDataFromKafka);
        this.scheduleTask(this::checkBatchTimeouts, this.getParameters().batchTimeoutMillis, TimeUnit.MILLISECONDS);
        this.getParameters().streamTimeoutMillis.ifPresent(timeout -> this.scheduleTask(() -> this.shutdownGracefully("Stream timeout reached"), (long)timeout, TimeUnit.MILLISECONDS));
        this.lastCommitMillis = System.currentTimeMillis();
        this.scheduleTask(this::checkCommitTimeout, this.getParameters().commitTimeoutMillis, TimeUnit.MILLISECONDS);
    }

    private void checkCommitTimeout() {
        long currentMillis = System.currentTimeMillis();
        boolean hasUncommitted = this.offsets.values().stream().filter(d -> !d.isCommitted()).findAny().isPresent();
        if (hasUncommitted) {
            long millisFromLastCommit = currentMillis - this.lastCommitMillis;
            if (millisFromLastCommit >= this.getParameters().commitTimeoutMillis) {
                this.shutdownGracefully("Commit timeout reached");
            } else {
                this.scheduleTask(this::checkCommitTimeout, this.getParameters().commitTimeoutMillis - millisFromLastCommit, TimeUnit.MILLISECONDS);
            }
        } else {
            this.scheduleTask(this::checkCommitTimeout, this.getParameters().commitTimeoutMillis, TimeUnit.MILLISECONDS);
        }
    }

    private void shutdownGracefully(String reason) {
        this.getLog().info("Shutting down gracefully. Reason: {}", (Object)reason);
        Map<Partition.PartitionKey, Long> uncommitted = this.offsets.entrySet().stream().filter(e -> !((PartitionData)e.getValue()).isCommitted()).collect(Collectors.toMap(Map.Entry::getKey, e -> ((PartitionData)e.getValue()).getSentOffset()));
        if (uncommitted.isEmpty()) {
            this.switchState(new CleanupState());
        } else {
            this.switchState(new ClosingState(uncommitted, this.lastCommitMillis));
        }
    }

    private void pollDataFromKafka() {
        if (this.kafkaConsumer == null) {
            throw new IllegalStateException("kafkaConsumer should not be null when calling pollDataFromKafka method");
        }
        if (this.isConnectionReady()) {
            if (this.kafkaConsumer.assignment().isEmpty() || this.pollPaused) {
                this.scheduleTask(this::pollDataFromKafka, this.getKafkaPollTimeout(), TimeUnit.MILLISECONDS);
                return;
            }
            ConsumerRecords records = this.kafkaConsumer.poll(this.getKafkaPollTimeout());
            if (!records.isEmpty()) {
                for (TopicPartition tp : records.partitions()) {
                    Partition.PartitionKey pk = new Partition.PartitionKey(tp.topic(), String.valueOf(tp.partition()));
                    Optional.ofNullable(this.offsets.get(pk)).ifPresent(pd -> records.records(tp).stream().forEach(record -> pd.addEventFromKafka(record.offset(), (String)record.value())));
                }
                this.addTask(this::streamToOutput);
            }
            this.addTask(this::pollDataFromKafka);
        } else {
            this.shutdownGracefully("Hila connection closed via crutch");
        }
    }

    private long getMessagesAllowedToSend() {
        long unconfirmed = this.offsets.values().stream().mapToLong(PartitionData::getUnconfirmed).sum();
        long allowDueWindowSize = (long)this.getParameters().windowSizeMessages - unconfirmed;
        return this.getParameters().getMessagesAllowedToSend(allowDueWindowSize, this.sentEvents);
    }

    private void checkBatchTimeouts() {
        this.streamToOutput();
        OptionalLong lastSent = this.offsets.values().stream().mapToLong(PartitionData::getLastSendMillis).min();
        long nextCall = lastSent.orElse(System.currentTimeMillis()) + this.getParameters().batchTimeoutMillis;
        long delta = nextCall - System.currentTimeMillis();
        if (delta > 0L) {
            this.scheduleTask(this::checkBatchTimeouts, delta, TimeUnit.MILLISECONDS);
        } else {
            this.getLog().debug("Probably acting too slow, stream timeouts are constantly rescheduled");
            this.addTask(this::checkBatchTimeouts);
        }
    }

    private void streamToOutput() {
        long currentTimeMillis = System.currentTimeMillis();
        int freeSlots = (int)this.getMessagesAllowedToSend();
        block0: for (Map.Entry<Partition.PartitionKey, PartitionData> e : this.offsets.entrySet()) {
            SortedMap<Long, String> toSend;
            while (null != (toSend = e.getValue().takeEventsToStream(currentTimeMillis, Math.min(this.getParameters().batchLimitEvents, freeSlots), this.getParameters().batchTimeoutMillis))) {
                this.flushData(e.getKey(), toSend);
                this.sentEvents += (long)toSend.size();
                if (toSend.isEmpty()) continue block0;
                freeSlots -= toSend.size();
            }
        }
        boolean bl = this.pollPaused = this.getMessagesAllowedToSend() <= 0L;
        if (!this.offsets.isEmpty() && this.getParameters().isKeepAliveLimitReached(this.offsets.values().stream().mapToInt(PartitionData::getKeepAliveInARow))) {
            this.shutdownGracefully("All partitions reached keepAlive limit");
        }
    }

    private void flushData(Partition.PartitionKey pk, SortedMap<Long, String> data) {
        String evt = EventStream.createStreamEvent(pk.partition, String.valueOf(this.offsets.get(pk).getSentOffset()), new ArrayList<String>(data.values()), Optional.empty());
        try {
            this.getOut().streamData(evt.getBytes(EventStream.UTF8));
        }
        catch (IOException e) {
            this.getLog().error("Failed to write data to output.", (Throwable)e);
            this.shutdownGracefully("Failed to write data to output");
        }
    }

    @Override
    public void onExit() {
        if (null != this.topologyChangeSubscription) {
            try {
                this.topologyChangeSubscription.cancel();
            }
            catch (RuntimeException ex) {
                this.getLog().warn("Failed to cancel topology subscription", (Throwable)ex);
            }
            finally {
                this.topologyChangeSubscription = null;
                new HashSet<Partition.PartitionKey>(this.offsets.keySet()).forEach(this::removeFromStreaming);
            }
        }
        if (null != this.kafkaConsumer) {
            try {
                this.kafkaConsumer.close();
            }
            finally {
                this.kafkaConsumer = null;
            }
        }
    }

    void topologyChanged() {
        if (null != this.topologyChangeSubscription) {
            this.topologyChangeSubscription.refresh();
        }
        this.reactOnTopologyChange();
    }

    private void reactOnTopologyChange() {
        this.getZk().runLocked(() -> {
            Partition[] assignedPartitions = (Partition[])Stream.of(this.getZk().listPartitions()).filter(p -> this.getSessionId().equals(p.getSession())).toArray(Partition[]::new);
            this.addTask(() -> this.refreshTopologyUnlocked(assignedPartitions));
        });
    }

    void refreshTopologyUnlocked(Partition[] assignedPartitions) {
        Map<Partition.PartitionKey, Partition> newAssigned = Stream.of(assignedPartitions).filter(p -> p.getState() == Partition.State.ASSIGNED).collect(Collectors.toMap(Partition::getKey, p -> p));
        Map<Partition.PartitionKey, Partition> newReassigning = Stream.of(assignedPartitions).filter(p -> p.getState() == Partition.State.REASSIGNING).collect(Collectors.toMap(Partition::getKey, p -> p));
        this.offsets.keySet().stream().filter(e -> !newAssigned.containsKey(e)).filter(e -> !newReassigning.containsKey(e)).collect(Collectors.toList()).forEach(this::removeFromStreaming);
        this.releasingPartitions.keySet().stream().filter(p -> !newReassigning.containsKey(p)).collect(Collectors.toList()).forEach(this.releasingPartitions::remove);
        if (!newReassigning.isEmpty()) {
            newReassigning.keySet().forEach(this::addPartitionToReassigned);
        }
        newAssigned.values().stream().filter(p -> !this.offsets.containsKey(p.getKey())).forEach(this::addToStreaming);
        this.reassignCommitted();
        this.reconfigureKafkaConsumer(false);
        this.logPartitionAssignment("Topology refreshed");
    }

    private void logPartitionAssignment(String reason) {
        if (this.getLog().isInfoEnabled()) {
            this.getLog().info("{}. Streaming partitions: [{}]. Reassigning partitions: [{}]", new Object[]{reason, this.offsets.keySet().stream().filter(p -> !this.releasingPartitions.containsKey(p)).map(Partition.PartitionKey::toString).collect(Collectors.joining(",")), this.releasingPartitions.keySet().stream().map(Partition.PartitionKey::toString).collect(Collectors.joining(", "))});
        }
    }

    private void addPartitionToReassigned(Partition.PartitionKey partitionKey) {
        long currentTime = System.currentTimeMillis();
        long barrier = currentTime + this.getParameters().commitTimeoutMillis;
        this.releasingPartitions.put(partitionKey, barrier);
        this.scheduleTask(() -> this.barrierOnRebalanceReached(partitionKey), this.getParameters().commitTimeoutMillis, TimeUnit.MILLISECONDS);
    }

    private void barrierOnRebalanceReached(Partition.PartitionKey pk) {
        if (!this.releasingPartitions.containsKey(pk)) {
            return;
        }
        this.getLog().info("Checking barrier to transfer partition {}", (Object)pk);
        long currentTime = System.currentTimeMillis();
        if (currentTime >= this.releasingPartitions.get(pk)) {
            this.shutdownGracefully("barrier on reassigning partition reached for " + pk + ", current time: " + currentTime + ", barrier: " + this.releasingPartitions.get(pk));
        } else {
            this.scheduleTask(() -> this.barrierOnRebalanceReached(pk), this.releasingPartitions.get(pk) - currentTime, TimeUnit.MILLISECONDS);
        }
    }

    private void reconfigureKafkaConsumer(boolean forceSeek) {
        Set currentNakadiAssignment;
        if (this.kafkaConsumer == null) {
            throw new IllegalStateException("kafkaConsumer should not be null when calling reconfigureKafkaConsumer method");
        }
        Set currentKafkaAssignment = this.kafkaConsumer.assignment().stream().map(tp -> new Partition.PartitionKey(tp.topic(), String.valueOf(tp.partition()))).collect(Collectors.toSet());
        if (!currentKafkaAssignment.equals(currentNakadiAssignment = this.offsets.keySet().stream().filter(o -> !this.releasingPartitions.containsKey(o)).collect(Collectors.toSet())) || forceSeek) {
            Map<Partition.PartitionKey, TopicPartition> kafkaKeys = currentNakadiAssignment.stream().collect(Collectors.toMap(k -> k, k -> new TopicPartition(k.topic, Integer.valueOf(k.partition).intValue())));
            this.kafkaConsumer.assign(new ArrayList<TopicPartition>(kafkaKeys.values()));
            this.kafkaConsumer.seekToBeginning(kafkaKeys.values().toArray(new TopicPartition[kafkaKeys.size()]));
            kafkaKeys.forEach((key, kafka) -> this.offsets.get(key).ensureDataAvailable(this.kafkaConsumer.position(kafka)));
            kafkaKeys.forEach((k, v) -> this.kafkaConsumer.seek(v, this.offsets.get(k).getSentOffset() + 1L));
            this.offsets.values().forEach(PartitionData::clearEvents);
        }
    }

    private void addToStreaming(Partition partition) {
        this.offsets.put(partition.getKey(), new PartitionData(this.getZk().subscribeForOffsetChanges(partition.getKey(), () -> this.addTask(() -> this.offsetChanged(partition.getKey()))), this.getZk().getOffset(partition.getKey()), LoggerFactory.getLogger((String)("subscription." + this.getSessionId() + "." + partition.getKey()))));
    }

    private void reassignCommitted() {
        List<Partition.PartitionKey> keysToRelease = this.releasingPartitions.keySet().stream().filter(pk -> !this.offsets.containsKey(pk) || this.offsets.get(pk).isCommitted()).collect(Collectors.toList());
        if (!keysToRelease.isEmpty()) {
            try {
                keysToRelease.forEach(this::removeFromStreaming);
            }
            finally {
                this.getZk().runLocked(() -> this.getZk().transfer(this.getSessionId(), keysToRelease));
            }
        }
    }

    void offsetChanged(Partition.PartitionKey key) {
        if (this.offsets.containsKey(key)) {
            PartitionData data = this.offsets.get(key);
            data.getSubscription().refresh();
            long offset = this.getZk().getOffset(key);
            PartitionData.CommitResult commitResult = data.onCommitOffset(offset);
            if (commitResult.seekOnKafka) {
                this.reconfigureKafkaConsumer(true);
            }
            if (commitResult.committedCount > 0L) {
                this.committedEvents += commitResult.committedCount;
                this.lastCommitMillis = System.currentTimeMillis();
                this.streamToOutput();
            }
            if (this.getParameters().isStreamLimitReached(this.committedEvents)) {
                this.shutdownGracefully("Stream limit in events reached: " + this.committedEvents);
            }
            if (this.releasingPartitions.containsKey(key) && data.isCommitted()) {
                this.reassignCommitted();
                this.logPartitionAssignment("New offset received for releasing partition " + key);
            }
        }
    }

    private void removeFromStreaming(Partition.PartitionKey key) {
        this.getLog().info("Removing partition {} from streaming", (Object)key);
        this.releasingPartitions.remove(key);
        PartitionData data = this.offsets.remove(key);
        if (null != data) {
            try {
                if (data.getUnconfirmed() > 0L) {
                    this.getLog().warn("Skipping commits: {}, commit={}, sent={}", new Object[]{key, data.getSentOffset() - data.getUnconfirmed(), data.getSentOffset()});
                }
                data.getSubscription().cancel();
            }
            catch (RuntimeException ex) {
                this.getLog().warn("Failed to cancel subscription, skipping exception", (Throwable)ex);
            }
        }
    }
}

