package org.zalando.opentracing.proxy;

import io.opentracing.Span;
import io.opentracing.tag.BooleanTag;
import io.opentracing.tag.StringTag;
import io.opentracing.tag.Tag;
import lombok.AllArgsConstructor;

import java.util.Map;
import java.util.Optional;

@AllArgsConstructor
final class ProxySpan extends ForwardingSpan {

    private final Span delegate;
    private final Options options;

    @Override
    protected Span delegate() {
        return delegate;
    }

    @Override
    protected Span create(final Span delegate) {
        return new ProxySpan(delegate, options);
    }

    @Override
    public Span setTag(final String key, final String value) {
        final Span span = super.setTag(key, value);
        options.tags().onTag(span, new StringTag(key), value);
        return span;
    }

    @Override
    public Span setTag(final String key, final boolean value) {
        final Span span = super.setTag(key, value);
        options.tags().onTag(span, new BooleanTag(key), value);
        return span;
    }

    @Override
    public Span setTag(final String key, final Number value) {
        final Span span = super.setTag(key, value);
        options.tags().onTag(span, new NumberTag(key), value);
        return span;
    }

    @Override
    public <T> Span setTag(final Tag<T> tag, final T value) {
        final Span span = super.setTag(tag, value);
        options.tags().onTag(span, tag, value);
        return span;
    }

    @Override
    public Span log(final Map<String, ?> fields) {
        final Span span = super.log(fields);
        options.logs().onLog(span, fields);
        return span;
    }

    @Override
    public Span log(final long timestamp, final Map<String, ?> fields) {
        final Span span = super.log(timestamp, fields);
        options.logs().onLog(span, timestamp, fields);
        return span;
    }

    @Override
    public Span log(final String event) {
        final Span span = super.log(event);
        options.logs().onLog(span, event);
        return span;
    }

    @Override
    public Span log(final long timestamp, final String event) {
        final Span span = super.log(timestamp, event);
        options.logs().onLog(span, timestamp, event);
        return span;
    }

    @Override
    public Span setBaggageItem(final String key, final String value) {
        final Span span = super.setBaggageItem(key, value);
        options.baggage().onBaggage(span, key, value);
        return span;
    }

    @Override
    public Span setOperationName(final String operationName) {
        return super.setOperationName(options.naming().rename(operationName));
    }

    @Override
    public void finish() {
        options.spans().onFinishing(this);
        super.finish();
        options.spans().onFinished(this);
    }

    @Override
    public void finish(final long finishMicros) {
        options.spans().onFinishing(this);
        super.finish(finishMicros);
        options.spans().onFinished(this);
    }

    static Span unwrap(final Span span) {
        return Optional.of(span)
                .filter(ProxySpan.class::isInstance)
                .map(ProxySpan.class::cast)
                .map(ProxySpan::delegate)
                .map(ProxySpan::unwrap) // recursive call
                .orElse(span);
    }

}
