/**
 * Copyright (C) 2015 Zalando SE (team-phrasers+oss@zalando.de)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zalando.jersey.gson.internal;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

import java.util.Iterator;
import java.util.ServiceLoader;

import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import javax.ws.rs.ext.ContextResolver;
import javax.ws.rs.ext.MessageBodyReader;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;
import javax.ws.rs.ext.Providers;

import org.zalando.jersey.gson.GsonProvider;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

/**
 * Provides JSON support backed by Google Gson for the JAX-RS runtime.
 *
 * <p>The Gson instance is tried to be loaded from a {@code ContextResolver}&lt;Gson&gt; instance or if that comes up empty
 * via a {@link ServiceLoader}&lt;org.zalando.jersey.gson.GsonProvider&gt;}.</p>
 *
 * @author  <a href="mailto:tom.wieczorek@zalando.de">Tom Wieczorek</a>
 * @author  <a href="mailto:christoph.berg@zalando.de">Christoph Berg</a>
 */
@Provider
@Produces(MediaType.APPLICATION_JSON)
@Consumes(MediaType.APPLICATION_JSON)
public class GsonJsonProvider implements MessageBodyWriter<Object>, MessageBodyReader<Object> {

    @Context
    private Providers providers;

    private Gson gson;

    private Charset charset = StandardCharsets.UTF_8;

    private Gson getGson() {
        if (gson != null) {
            return gson;
        }

        final ContextResolver<Gson> contextResolver = providers.getContextResolver(Gson.class,
                MediaType.APPLICATION_JSON_TYPE);

        if (contextResolver != null) {
            gson = contextResolver.getContext(Gson.class);
        } else {
            final Iterator<GsonProvider> providers = ServiceLoader.load(GsonProvider.class).iterator();
            if (providers.hasNext()) {
                gson = providers.next().get();
            }
        }

        if (gson == null) {
            gson = new Gson();
        }

        return gson;
    }

    @Override
    public boolean isReadable(final Class<?> type, final Type genericType, final Annotation[] annotations,
            final MediaType mediaType) {
        return true;
    }

    @Override
    public Object readFrom(final Class<Object> type, final Type genericType, final Annotation[] annotations,
            final MediaType mediaType, final MultivaluedMap<String, String> httpHeaders, final InputStream entityStream)
        throws IOException {
        final Type jsonType = type.equals(genericType) ? type : genericType;
        try(final Reader reader = new InputStreamReader(entityStream, charset)) {
            return getGson().fromJson(reader, jsonType);
        } catch (final JsonSyntaxException e) {
            throw new WebApplicationException(e.getMessage(), e, Response.Status.BAD_REQUEST);
        }
    }

    @Override
    public boolean isWriteable(final Class<?> type, final Type genericType, final Annotation[] annotations,
            final MediaType mediaType) {
        return true;
    }

    @Override
    public long getSize(final Object object, final Class<?> type, final Type genericType,
            final Annotation[] annotations, final MediaType mediaType) {
        return -1;
    }

    @Override
    public void writeTo(final Object object, final Class<?> type, final Type genericType,
            final Annotation[] annotations, final MediaType mediaType, final MultivaluedMap<String, Object> httpHeaders,
            final OutputStream entityStream) throws IOException, WebApplicationException {
        final Type jsonType = type.equals(genericType) ? type : genericType;
        try(final Writer writer = new OutputStreamWriter(entityStream, charset)) {
            getGson().toJson(object, jsonType, writer);
        }
    }
}
