package org.zalando.straw;

import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


public class Straw {

    public static final long OFFSET_BEGIN = -1;

    public static final class Cursor {

        static Cursor extract(String line) {
            Scanner scanner = new Scanner(line);
            int partition = Integer.parseInt(scanner.findInLine("\\d+"));
            long offset = Long.parseLong(scanner.findInLine("\\d+"));
            return new Cursor(partition, offset);
        }

        public final int partition;
        public final long offset;

        Cursor(int partition, long offset) {
            this.partition = partition;
            this.offset = offset;
        }

        @Override public String toString() {
            return String.format("{\"partition\":\"%d\",\"offset\":\"%s\"}", partition, offset());
        }

        private String offset() {
            return offset == OFFSET_BEGIN ? "BEGIN" : Long.toString(offset);
        }
    }

    private final ExecutorService _executor = Executors.newSingleThreadExecutor();
    private final URL _url;
    private final Map<Integer, Long> _cursors;

    public Straw(URL url, Map<Integer, Long> cursors) {
        _url = url;
        _cursors = new HashMap(cursors);
    }

    public void start() {
        _executor.submit(() -> { while (true) fetchStream(); });
    }

    protected Map<Integer, Long> getCursors() {
        return Collections.unmodifiableMap(_cursors);
    }

    protected String loadToken() throws Exception {
        return System.getenv("TOKEN");
    }

    protected void storeCursor(Cursor cursor) throws Exception {
        logDebug("storeCursor: " + cursor);
    }

    protected void handleEvents(String json) throws Exception {
        logDebug("handleEvents: " + json);
    }

    protected void logDebug(String message) {
        System.out.println("DEBUG: " + message);
    }

    protected void logInfo(String message) {
        System.out.println("INFO: " + message);
    }

    protected void logError(String message) {
        System.out.println("ERROR: " + message);
    }

    private void fetchStream() {
        logInfo("fetchStream: " + cursorString());
        try {
            SSLSocket socket = (SSLSocket) SSLSocketFactory.getDefault().createSocket(_url.getHost(), 443);
            try {
                sendRequest(socket);
                BufferedReader in = new BufferedReader(new InputStreamReader(socket.getInputStream(), "UTF-8"));
                int statusCode = readHeaders(in);
                if (statusCode != 200) {
                    in.readLine(); // skip number of payload bytes
                    throw new Exception(statusCode + ": " + in.readLine());
                }
                int i = 1;
                String line;
                while ((line = in.readLine()) != null) {
                    // chunked encoding, lines alternating between:
                    // 1) newline 2) number of payload bytes 3) payload
                    i++;
                    if ((i % 3) == 0) handleBatch(line.trim());
                }
            } finally {
                socket.close();
            }
        } catch (Exception e) {
            logError(e.getMessage());
            tryToSleep(2000);
        }
    }

    private void handleBatch(String line) throws Exception {
        if (!line.isEmpty()) {
            Cursor cursor = Cursor.extract(line);
            if (cursor.offset > _cursors.get(cursor.partition)) {
                handleEvents(line);
                storeCursor(cursor);
                // no exception, so we can update _cursors
                _cursors.put(cursor.partition, cursor.offset);
            } else {
                // just a keep-alive, ignore
            }
        }
    }

    private void sendRequest(SSLSocket socket) throws Exception {
        socket.startHandshake();
        PrintWriter out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(socket.getOutputStream())));
        out.println("GET " + requestPath() + " HTTP/1.1");
        out.println("Host: " + _url.getHost());
        out.println("Authorization: Bearer " + loadToken());
        out.println("X-Nakadi-Cursors: " + cursorString());
        out.println("User-Agent: straw");
        out.println("Accept: */*");
        out.println();
        out.flush();
    }

    private String requestPath() {
        return _url.getQuery() == null ? _url.getPath() : _url.getPath() + "?" + _url.getQuery();
    }

    private String cursorString() {
        List<Cursor> result = new ArrayList();
        for (int partition : _cursors.keySet()) {
            result.add(new Cursor(partition, _cursors.get(partition)));
        }
        return Arrays.toString(result.toArray());
    }

    private int readHeaders(BufferedReader out) throws IOException {
        String line;
        int statusCode = -1;
        while ((line = out.readLine()) != null) {
            if (statusCode == -1) {
                statusCode = Integer.parseInt(line.split("\\s")[1]);
            } else if (line.trim().isEmpty()) {
                 break;
            }
        }
        return statusCode;
    }

    private static void tryToSleep(int millis) {
        try { Thread.sleep(millis); } catch (InterruptedException ignored) {}
    }
}
