package org.zalando.straw;

import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.URL;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


public class Straw {

    public static final long OFFSET_BEGIN = -1;

    public static final class Cursor {

        static Cursor extract(String line) throws Exception {
            try {
                Scanner scanner = new Scanner(line);
                int partition = Integer.parseInt(scanner.findInLine("\\d+"));
                long offset = Long.parseLong(scanner.findInLine("\\d+"));
                return new Cursor(partition, offset);
            } catch (NumberFormatException e) {
                throw new Exception(line);
            }
        }

        public final int partition;
        public final long offset;

        Cursor(int partition, long offset) {
            this.partition = partition;
            this.offset = offset;
        }

        @Override public String toString() {
            return String.format("{\"partition\":\"%d\",\"offset\":\"%s\"}", partition, offset());
        }

        private String offset() {
            return offset == OFFSET_BEGIN ? "BEGIN" : Long.toString(offset);
        }
    }

    private final ExecutorService _executor = Executors.newSingleThreadExecutor();
    private final URL _url;
    private final Map<Integer, Long> _cursors;
    private final boolean _allPartitions;

    public Straw(URL url, Map<Integer, Long> cursors) {
        _url = url;
        _cursors = new HashMap(cursors);
        _allPartitions = cursors.isEmpty();
    }

    public void start() {
        _executor.submit(() -> { while (true) fetchStream(); });
    }

    protected String loadToken() throws Exception {
        return System.getenv("TOKEN");
    }

    protected void handleEvents(String json) throws Exception {
        logDebug("handleEvents: " + json);
    }

    protected void logDebug(String message) {
        System.out.println("DEBUG: " + message);
    }

    protected void logInfo(String message) {
        System.out.println("INFO: " + message);
    }

    protected void logError(String message) {
        System.out.println("ERROR: " + message);
    }

    private void fetchStream() {
        logInfo("fetchStream: " + (_allPartitions ? "all partitions (END)" : cursorString()));
        try {
            SSLSocket socket = (SSLSocket) SSLSocketFactory.getDefault().createSocket(_url.getHost(), getPort());
            try {
                sendRequest(socket);
                BufferedReader in = new BufferedReader(new InputStreamReader(socket.getInputStream(), "UTF-8"));
                int statusCode = readHeaders(in);
                if (statusCode != 200) {
                    in.readLine(); // skip number of payload bytes
                    throw new Exception(statusCode + ": " + in.readLine());
                }
                // simple state machine to read chunked encoding.
                // each line can be SIZE, DATA or EMPTY.
                // a single nakadi batch can span multiple chunks, for example:
                // SIZE DATA SIZE DATA SIZE DATA EMPTY
                StringBuilder batch = new StringBuilder();
                boolean isData = false;
                String line;
                while ((line = in.readLine()) != null) {
                    if (isData) {
                        batch.append(line);
                        isData = false;
                    } else {
                        if (line.isEmpty()) {
                            handleBatch(batch.toString());
                            batch.setLength(0);
                        } else {
                            isData = true;
                        }
                    }
                }
            } finally {
                socket.close();
            }
        } catch (Exception e) {
            logError(e.getMessage());
            tryToSleep(2000);
        }
    }

    private void handleBatch(String line) throws Exception {
        if (!line.isEmpty()) {
            Cursor cursor = Cursor.extract(line);
            handleEvents(line);
            // no exception, so we can update _cursors
            if (cursor.offset > _cursors.getOrDefault(cursor.partition, OFFSET_BEGIN)) {
                _cursors.put(cursor.partition, cursor.offset);
            } 
        }
    }

    private void sendRequest(SSLSocket socket) throws Exception {
        socket.startHandshake();
        PrintWriter out = new PrintWriter(new BufferedWriter(new OutputStreamWriter(socket.getOutputStream())));
        out.println("GET " + requestPath() + " HTTP/1.1");
        out.println("Host: " + _url.getHost());
        out.println("User-Agent: straw");
        if (!_allPartitions) {
            out.println("X-Nakadi-Cursors: " + cursorString());
        }
        out.println("Authorization: Bearer " + loadToken().trim());
        out.println();
        out.flush();
    }

    private int getPort() {
        return _url.getPort() == -1 ? 443 : _url.getPort();
    }

    private String requestPath() {
        return _url.getQuery() == null ? _url.getPath() : _url.getPath() + "?" + _url.getQuery();
    }

    private String cursorString() {
        List<Cursor> result = new ArrayList();
        for (int partition : _cursors.keySet()) {
            result.add(new Cursor(partition, _cursors.get(partition)));
        }
        return Arrays.toString(result.toArray());
    }

    private int readHeaders(BufferedReader out) throws IOException {
        String line;
        int statusCode = -1;
        while ((line = out.readLine()) != null) {
            if (statusCode == -1) {
                statusCode = Integer.parseInt(line.split("\\s")[1]);
            } else if (line.trim().isEmpty()) {
                 break;
            }
        }
        return statusCode;
    }

    private static void tryToSleep(int millis) {
        try { Thread.sleep(millis); } catch (InterruptedException ignored) {}
    }
}
