/*
 * This program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v2.0 which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-v20.html
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Copyright Contributors to the Zowe Project.
 */
package zowe.client.sdk.zostso.method;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import zowe.client.sdk.core.ZosConnection;
import zowe.client.sdk.rest.exception.ZosmfRequestException;
import zowe.client.sdk.utility.ValidateUtils;
import zowe.client.sdk.zostso.TsoConstants;
import zowe.client.sdk.zostso.input.StartTsoInputData;
import zowe.client.sdk.zostso.service.TsoReplyService;
import zowe.client.sdk.zostso.service.TsoSendService;
import zowe.client.sdk.zostso.service.TsoStartService;
import zowe.client.sdk.zostso.service.TsoStopService;

import java.util.ArrayList;
import java.util.List;

/**
 * Issue tso command via z/OSMF restful api
 *
 * @author Frank Giordano
 * @version 5.0
 */
public class IssueTso {

    private final List<String> msgLst = new ArrayList<>();
    private final List<String> promptLst = new ArrayList<>();
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final ZosConnection connection;
    private String accountNumber;
    private TsoStartService tsoStartService;
    private TsoStopService tsoStopService;
    private TsoSendService tsoSendService;
    private TsoReplyService tsoReplyService;

    /**
     * IssueTso constructor
     *
     * @param connection    ZosConnection object
     * @param accountNumber account number for tso processing
     * @author Frank Giordano
     */
    public IssueTso(final ZosConnection connection, final String accountNumber) {
        ValidateUtils.checkNullParameter(connection == null, "connection is null");
        ValidateUtils.checkIllegalParameter(accountNumber, "accountNumber");
        this.connection = connection;
        this.accountNumber = accountNumber;
    }

    /**
     * Alternative IssueTso constructor with ZoweRequest object. This is mainly used for internal code unit
     * testing with mockito, and it is not recommended to be used by the larger community.
     * <p>
     * This constructor is package-private
     *
     * @param connection      for connection information, see ZosConnection object
     * @param accountNumber   account number for tso processing
     * @param tsoStartService TsoStartService for mocking
     * @param tsoStopService  TsoStopService for mocking
     * @param tsoSendService  TsoSendService for mocking
     * @param tsoReplyService TsoReplyService for mocking
     * @author Frank Giordano
     */
    IssueTso(final ZosConnection connection,
             final String accountNumber,
             final TsoStartService tsoStartService,
             final TsoStopService tsoStopService,
             final TsoSendService tsoSendService,
             final TsoReplyService tsoReplyService) {
        ValidateUtils.checkNullParameter(connection == null, "connection is null");
        this.connection = connection;
        this.accountNumber = accountNumber;
        this.tsoStartService = tsoStartService;
        this.tsoStopService = tsoStopService;
        this.tsoSendService = tsoSendService;
        this.tsoReplyService = tsoReplyService;
    }

    /**
     * Issue TSO command API call to process the given command via z/OSMF restful api
     *
     * @param command tso command string
     * @return list of all tso returned messages
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    public List<String> issueCommand(final String command) throws ZosmfRequestException {
        ValidateUtils.checkIllegalParameter(command, "command");
        return this.issueCommand(command, null);
    }

    /**
     * Issue TSO command API call to process the given command via z/OSMF restful with given custom
     * parameters for the start TSO session call
     *
     * @param command   tso command string
     * @param inputData start TSO request inputs parameters, see StartTsoInputData
     * @return list of all tso returned messages
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    public List<String> issueCommand(final String command, final StartTsoInputData inputData)
            throws ZosmfRequestException {
        ValidateUtils.checkIllegalParameter(command, "command");
        this.msgLst.clear();
        this.promptLst.clear();

        final String sessionId = this.startTso(inputData);

        String responseStr = this.sendTsoCommand(sessionId, command);
        JsonNode tsoData = this.getJsonNode(responseStr).get("tsoData");
        this.processTsoData(tsoData);

        while (promptLst.isEmpty()) {
            responseStr = this.sendTsoForReply(sessionId);
            tsoData = this.getJsonNode(responseStr).get("tsoData");
            this.processTsoData(tsoData);
        }

        this.stopTso(sessionId);
        return msgLst;
    }

    /**
     * Make the first TSO request to start the TSO session and retrieve its session id (servletKey).
     *
     * @param inputData start TSO request inputs parameters, see StartTsoInputData
     * @return string value representing the session id (servletKey)
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    private String startTso(StartTsoInputData inputData) throws ZosmfRequestException {
        if (tsoStartService == null) {
            tsoStartService = new TsoStartService(connection);
        }
        if (inputData == null) {
            inputData = new StartTsoInputData();
        }
        inputData.setAccount(accountNumber);
        return tsoStartService.startTso(inputData);
    }

    /**
     * Make the second request to send TSO the command to perform via z/OSMF
     *
     * @param sessionId servletKey id retrieve from start TSO request
     * @param command   tso command
     * @return response string representing the returned request payload
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    private String sendTsoCommand(final String sessionId, final String command) throws ZosmfRequestException {
        if (tsoSendService == null) {
            tsoSendService = new TsoSendService(connection);
        }
        return tsoSendService.sendCommand(sessionId, command);
    }

    /**
     * Send a request to z/OSMF TSO for additional TSO message data
     *
     * @param sessionId servletKey id retrieve from start TSO request
     * @return response string representing the returned request payload
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    private String sendTsoForReply(final String sessionId) throws ZosmfRequestException {
        if (tsoReplyService == null) {
            tsoReplyService = new TsoReplyService(connection);
        }
        return tsoReplyService.reply(sessionId);
    }

    /**
     * Stop the TSO session by session id (servletKey)
     *
     * @param sessionId servletKey id retrieve from start TSO request
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    private void stopTso(final String sessionId) throws ZosmfRequestException {
        if (tsoStopService == null) {
            tsoStopService = new TsoStopService(connection);
        }
        tsoStopService.stopTso(sessionId);
    }

    /**
     * Transform the JSON response payload for its tso message types
     *
     * @param tsoData JsonNode object
     * @author Frank Giordano
     */
    private void processTsoData(final JsonNode tsoData) {
        tsoData.forEach(tsoDataItem -> {
            try {
                if (tsoDataItem.get(TsoConstants.TSO_MESSAGE) != null) {
                    this.msgLst.add(tsoDataItem.get("TSO MESSAGE").get("DATA").asText());
                }
                if (tsoDataItem.get(TsoConstants.TSO_PROMPT) != null) {
                    this.promptLst.add(tsoDataItem.get("TSO PROMPT").get("HIDDEN").asText());
                }
            } catch (Exception ignored) {
            }
        });
    }

    /**
     * Transform a response string representing a JSON returned payload from a tso call into a JsonNode to
     * be used for parsing the response.
     *
     * @param responseStr response string
     * @return JsonNode object
     * @throws ZosmfRequestException request error state
     * @author Frank Giordano
     */
    private JsonNode getJsonNode(final String responseStr) throws ZosmfRequestException {
        final JsonNode rootNode;
        try {
            rootNode = objectMapper.readTree(responseStr);
        } catch (JsonProcessingException e) {
            throw new ZosmfRequestException(TsoConstants.SEND_TSO_FAIL_MSG + " Response: " + e.getMessage());
        }
        return rootNode;
    }

    /**
     * Allow the ability to change account number
     *
     * @param accountNumber string value
     * @author Frank Giordano
     */
    public void setAccountNumber(String accountNumber) {
        this.accountNumber = accountNumber;
    }

}
