/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package pl.gsmservice.gateway.utils;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Generic streaming parser that handles byte buffer management and delegates
 * format-specific logic to a StreamContentProcessor.
 */
public final class StreamingParser<T> {

    /**
    * Information about a found boundary in byte data
    */
    static class BoundaryInfo {
        public final int position;
        public final int delimiterLength;

        public BoundaryInfo(int position, int delimiterLength) {
            this.position = position;
            this.delimiterLength = delimiterLength;
        }
    }

    /**
    * Interface for format-specific parsing logic
    */
    interface StreamContentProcessor<T> {
        /**
        * Find the next boundary in the byte buffer
        * @return boundary info, or position -1 if no boundary found
        */
        BoundaryInfo findBoundary(byte[] data, int limit);

        /**
        * Process extracted content and return the parsed result
        * @param content the extracted content (without boundary delimiters)
        * @return parsed result, or empty if content should be skipped
        */
        Optional<T> processContent(String content);

        /**
        * Sanitize content text (e.g., handle line endings, BOM, etc.)
        * @param rawContent the raw extracted content
        * @param isFirst whether this is the first content processed
        * @return sanitized content
        */
        default String sanitizeContent(String rawContent, boolean isFirst) {
            return rawContent.replace("\r\n", "\n").replace("\r", "\n");
        }
    }

    private final StreamContentProcessor<T> processor;
    private ByteBuffer byteBuffer = ByteBuffer.allocate(8192);
    private boolean first = true;

    StreamingParser(StreamContentProcessor<T> processor) {
        this.processor = processor;
    }

    /**
     * Add ByteBuffer data to the parser buffer and extract any complete items.
     *
     * @param inputBuffer byte data to add (will not be modified)
     * @return next complete parsed result if one becomes available
     */
    public Optional<T> add(ByteBuffer inputBuffer) {
        if (inputBuffer == null || !inputBuffer.hasRemaining()) {
            return extractNextFromBytes();
        }
        // Ensure we have enough capacity
        if (byteBuffer.remaining() < inputBuffer.remaining()) {
            byteBuffer = expandByteBuffer(byteBuffer.position() + inputBuffer.remaining());
        }
        // Append new data
        byteBuffer.put(inputBuffer.slice());
        return extractNextFromBytes();
    }

    /**
     * Extract any remaining partial content when stream ends.
     *
     * @return final parsed result if there was incomplete data in the buffer
     */
    public Optional<T> finish() {
        if (byteBuffer.position() > 0) {
            byte[] remainingBytes = new byte[byteBuffer.position()];
            byteBuffer.flip();
            byteBuffer.get(remainingBytes);
            byteBuffer.clear();
            String content = processor.sanitizeContent(new String(remainingBytes, StandardCharsets.UTF_8), first);
            return processor.processContent(content);
        }
        return Optional.empty();
    }

    /**
     * Check if there are additional complete items in the buffer.
     *
     * @return next complete parsed result from buffer if available
     */
    public Optional<T> next() {
        return extractNextFromBytes();
    }

    /**
     * Check if parser has any buffered data
     */
    public boolean hasBufferedData() {
        return byteBuffer.position() > 0;
    }

    private Optional<T> extractNextFromBytes() {
        if (byteBuffer.position() == 0) {
            return Optional.empty();
        }
        // Find boundary directly in bytes
        BoundaryInfo boundary = processor.findBoundary(byteBuffer.array(), byteBuffer.position());
        if (boundary.position == -1) {
            return Optional.empty();
        }
        // Extract content bytes without copying the entire buffer
        byte[] contentBytes = new byte[boundary.position];
        byteBuffer.flip();
        byteBuffer.get(contentBytes, 0, boundary.position);
        // Compact buffer to remove processed content + delimiter
        byteBuffer.position(boundary.position + boundary.delimiterLength);
        byteBuffer.compact();
        String content = processor.sanitizeContent(new String(contentBytes, StandardCharsets.UTF_8), first);
        if (first) {
            first = false;
        }
        Optional<T> result = processor.processContent(content);
        if (result.isPresent()) {
            return result;
        }
        // Check for additional items if this one was skipped
        return extractNextFromBytes();
    }

    private ByteBuffer expandByteBuffer(int newCapacity) {
        ByteBuffer newBuffer = ByteBuffer.allocate(Math.max(newCapacity, byteBuffer.capacity() * 2));
        byteBuffer.flip();
        newBuffer.put(byteBuffer);
        return newBuffer;
    }

    /**
     * Check if a byte pattern matches at a specific position
     */
    public static boolean matchesPattern(byte[] data, int pos, int limit, byte... pattern) {
        if (pos + pattern.length > limit) {
            return false;
        }
        for (int i = 0; i < pattern.length; i++) {
            if (data[pos + i] != pattern[i]) {
                return false;
            }
        }
        return true;
    }

    // ===== JSON Lines Content Processor =====
    
    /**
     * JSON Lines content processor implementation
     */
    private static class JsonLContentProcessor implements StreamContentProcessor<String> {
        // Line boundary patterns
        private static final byte CR = '\r';
        private static final byte LF = '\n';
        private static final byte[] CRLF = {CR, LF}; // \r\n
        private static final byte[] LF_ONLY = {LF}; // \n

        @Override
        public BoundaryInfo findBoundary(byte[] data, int limit) {
            for (int i = 0; i < limit; i++) {
                // Check for CRLF first (longer pattern)
                if (matchesPattern(data, i, limit, CRLF)) {
                    return new BoundaryInfo(i, CRLF.length);
                }
                // Check for LF only
                if (matchesPattern(data, i, limit, LF_ONLY)) {
                    return new BoundaryInfo(i, LF_ONLY.length);
                }
            }
            return new BoundaryInfo(-1, 0);
        }

        @Override
        public Optional<String> processContent(String content) {
            String trimmed = content.trim();
            // Return non-empty JSON lines
            return trimmed.isEmpty() ? Optional.empty() : Optional.of(trimmed);
        }
    }

    // ===== SSE Content Processor =====
    
    /**
     * SSE content processor implementation
     */
    private static class SSEContentProcessor implements StreamContentProcessor<EventStreamMessage> {
        private static final String BYTE_ORDER_MARK = "\uFEFF";
        private static final Pattern LINE_PATTERN = Pattern.compile("^([a-zA-Z]+): ?(.*)$");
        private static final char LINEFEED = '\n';
        // Message boundary patterns
        private static final byte CR = '\r';
        private static final byte LF = '\n';
        private static final byte[] CRLF_CRLF = {CR, LF, CR, LF}; // \r\n\r\n
        private static final byte[] CRLF_LF = {CR, LF, LF}; // \r\n\n
        private static final byte[] LF_CRLF = {LF, CR, LF}; // \n\r\n
        private static final byte[] LF_LF = {LF, LF}; // \n\n

        @Override
        public BoundaryInfo findBoundary(byte[] data, int limit) {
            for (int i = 0; i < limit; i++) {
                // Need at least 2 bytes for any boundary pattern
                if (i + 1 >= limit) {
                    continue;
                }
                // Check longest patterns first to avoid partial matches
                if (matchesPattern(data, i, limit, CRLF_CRLF)) {
                    return new BoundaryInfo(i, CRLF_CRLF.length);
                }
                if (matchesPattern(data, i, limit, CRLF_LF)) {
                    return new BoundaryInfo(i, CRLF_LF.length);
                }
                if (matchesPattern(data, i, limit, LF_CRLF)) {
                    return new BoundaryInfo(i, LF_CRLF.length);
                }
                if (matchesPattern(data, i, limit, LF_LF)) {
                    return new BoundaryInfo(i, LF_LF.length);
                }
            }
            return new BoundaryInfo(-1, 0);
        }

        @Override
        public Optional<EventStreamMessage> processContent(String content) {
            if (content.trim().isEmpty()) {
                return Optional.empty();
            }
            return Optional.of(parseMessage(content));
        }

        @Override
        public String sanitizeContent(String rawContent, boolean isFirst) {
            String sanitized = rawContent.replace("\r\n", "\n").replace("\r", "\n");
            if (isFirst && sanitized.startsWith(BYTE_ORDER_MARK)) {
                sanitized = sanitized.substring(BYTE_ORDER_MARK.length());
            }
            return sanitized;
        }

        private EventStreamMessage parseMessage(String text) {
            String[] lines = text.split("\n");
            Optional<String> event = Optional.empty();
            Optional<String> id = Optional.empty();
            Optional<Integer> retryMs = Optional.empty();
            StringBuilder data = new StringBuilder();
            boolean firstData = true;
            for (String line : lines) {
                // Skip comment lines
                if (line.startsWith(":")) {
                    continue;
                }
                Matcher m = LINE_PATTERN.matcher(line);
                if (m.find()) {
                    String key = m.group(1).toLowerCase();
                    String value = m.group(2);
                    switch (key) {
                        case "event":
                            event = Optional.of(value);
                            break;
                        case "id":
                            id = Optional.of(value);
                            break;
                        case "retry":
                            try {
                                retryMs = Optional.of(Integer.parseInt(value));
                            } catch (NumberFormatException e) {
                                // ignore invalid retry values
                            }
                            break;
                        case "data":
                            if (!firstData) {
                                data.append(LINEFEED);
                            }
                            firstData = false;
                            data.append(value);
                            break;
                        // ignore unknown fields
                    }
                }
            }
            return new EventStreamMessage(event, id, retryMs, data.toString());
        }
    }

    // ===== Factory Methods =====

    /**
     * Create a streaming parser for JSON Lines format
     */
    public static StreamingParser<String> forJsonLines() {
        return new StreamingParser<>(new JsonLContentProcessor());
    }

    /**
     * Create a streaming parser for SSE format
     */
    public static StreamingParser<EventStreamMessage> forSSE() {
        return new StreamingParser<>(new SSEContentProcessor());
    }
}
