/*
 * Decompiled with CFR 0.152.
 */
package pl.gsmservice.gateway.utils;

import java.io.IOException;
import java.net.ConnectException;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import pl.gsmservice.gateway.utils.AsyncRetryableException;
import pl.gsmservice.gateway.utils.BackoffStrategy;
import pl.gsmservice.gateway.utils.Blob;
import pl.gsmservice.gateway.utils.NonRetryableException;
import pl.gsmservice.gateway.utils.RetryConfig;
import pl.gsmservice.gateway.utils.Utils;

public class AsyncRetries {
    private final RetryConfig retryConfig;
    private final List<String> retriableStatusCodes;
    private final ScheduledExecutorService scheduler;

    private AsyncRetries(RetryConfig retryConfig, List<String> retriableStatusCodes, ScheduledExecutorService scheduler) {
        Utils.checkNotNull(retryConfig, "retryConfig");
        Utils.checkNotNull(retriableStatusCodes, "statusCodes");
        if (retriableStatusCodes.isEmpty()) {
            throw new IllegalArgumentException("statusCodes list cannot be empty");
        }
        this.retryConfig = retryConfig;
        this.retriableStatusCodes = retriableStatusCodes;
        this.scheduler = scheduler;
    }

    public CompletableFuture<HttpResponse<Blob>> retry(Supplier<CompletableFuture<HttpResponse<Blob>>> task) {
        switch (this.retryConfig.strategy()) {
            case BACKOFF: {
                CompletableFuture<HttpResponse<Blob>> future = new CompletableFuture<HttpResponse<Blob>>();
                BackoffStrategy backoff = this.retryConfig.backoff().orElseThrow(() -> new IllegalArgumentException("Backoff strategy is not defined"));
                this.attempt(task, future, backoff, new State(0L, Instant.now()));
                return future;
            }
            case NONE: {
                return task.get();
            }
        }
        throw new IllegalArgumentException("Unsupported retry strategy: " + String.valueOf((Object)this.retryConfig.strategy()));
    }

    private <T> void attempt(Supplier<CompletableFuture<HttpResponse<Blob>>> task, CompletableFuture<HttpResponse<Blob>> result, BackoffStrategy backoff, State state) {
        task.get().whenComplete((response, throwable) -> {
            Throwable e;
            if (throwable == null) {
                boolean matched = this.retriableStatusCodes.stream().anyMatch(pattern -> Utils.statusCodeMatches(response.statusCode(), pattern));
                if (matched) {
                    this.maybeRetry(task, result, backoff, state, new AsyncRetryableException((HttpResponse<Blob>)response));
                    return;
                }
                result.complete((HttpResponse<Blob>)response);
                return;
            }
            Throwable throwable2 = e = throwable instanceof CompletionException ? throwable.getCause() : throwable;
            if (e instanceof AsyncRetryableException) {
                this.maybeRetry(task, result, backoff, state, e);
                return;
            }
            if (e instanceof IOException && this.shouldRetryIOException(e, backoff)) {
                this.maybeRetry(task, result, backoff, state, e);
                return;
            }
            result.completeExceptionally(new NonRetryableException(e));
        });
    }

    private boolean shouldRetryIOException(Throwable e, BackoffStrategy backoff) {
        if (e instanceof ConnectException && backoff.retryConnectError()) {
            return true;
        }
        String message = e.getMessage();
        if (message == null) {
            return false;
        }
        return message.contains("Connect timed out") && backoff.retryConnectError() || message.contains("Read timed out") && backoff.retryReadTimeoutError();
    }

    private void maybeRetry(Supplier<CompletableFuture<HttpResponse<Blob>>> task, CompletableFuture<HttpResponse<Blob>> result, BackoffStrategy backoff, State state, Throwable e) {
        Duration timeSinceStart = Duration.between(state.startedAt(), Instant.now());
        if (timeSinceStart.toMillis() > backoff.maxElapsedTimeMs()) {
            if (e instanceof AsyncRetryableException) {
                result.complete(((AsyncRetryableException)e).response());
                return;
            }
            result.completeExceptionally(e);
            return;
        }
        double intervalMs = (double)backoff.initialIntervalMs() * Math.pow(backoff.baseFactor(), state.count());
        double jitterMs = backoff.jitterFactor() * intervalMs;
        intervalMs = intervalMs - jitterMs + Math.random() * (2.0 * jitterMs + 1.0);
        intervalMs = Math.min(intervalMs, (double)backoff.maxIntervalMs());
        this.scheduler.schedule(() -> this.attempt(task, result, backoff, state.countAttempt()), (long)intervalMs, TimeUnit.MILLISECONDS);
    }

    public void shutdown() {
        this.scheduler.shutdown();
    }

    public static Builder builder() {
        return new Builder();
    }

    private static class State {
        private long attempt;
        private final Instant startedAt;

        public State(long attempt, Instant startedAt) {
            this.attempt = attempt;
            this.startedAt = startedAt;
        }

        public long count() {
            return this.attempt;
        }

        public Instant startedAt() {
            return this.startedAt;
        }

        public State countAttempt() {
            ++this.attempt;
            return this;
        }
    }

    public static final class Builder {
        private RetryConfig retryConfig;
        private List<String> statusCodes;
        private ScheduledExecutorService scheduler;

        private Builder() {
        }

        public Builder retryConfig(RetryConfig retryConfig) {
            Utils.checkNotNull(retryConfig, "retryConfig");
            this.retryConfig = retryConfig;
            return this;
        }

        public Builder statusCodes(List<String> statusCodes) {
            Utils.checkNotNull(statusCodes, "statusCodes");
            if (statusCodes.isEmpty()) {
                throw new IllegalArgumentException("statusCodes list cannot be empty");
            }
            this.statusCodes = statusCodes;
            return this;
        }

        public Builder scheduler(ScheduledExecutorService scheduler) {
            Utils.checkNotNull(scheduler, "scheduler");
            this.scheduler = scheduler;
            return this;
        }

        public AsyncRetries build() {
            return new AsyncRetries(this.retryConfig, this.statusCodes, this.scheduler);
        }
    }
}

