package pl.michal.grzesiak.criticizer;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

public class Criticizer {

    private final ClassesProvider classesProvider;
    private final ValidationAnnotationChecker validationAnnotationChecker;
    private final ReferenceValidator referenceValidator;

    Criticizer(ClassesProvider classesProvider,
               ValidationAnnotationChecker validationAnnotationChecker, ReferenceValidator referenceValidator) {
        this.classesProvider = classesProvider;
        this.validationAnnotationChecker = validationAnnotationChecker;
        this.referenceValidator = referenceValidator;
    }

    public List<IncorrectValidation> checkAllValidationsIn(String packageName) {
        return checkAllValidationsIn(packageName, "");
    }

    public List<IncorrectValidation> checkAllValidationsIn(String packageName, String classNameSuffix) {
        List<Class<?>> classes = classesProvider.getClasses(packageName, classNameSuffix);

        return findIncorrectValidationsIn(classes);
    }

    private List<IncorrectValidation> findIncorrectValidationsIn(List<Class<?>> classes) {
        List<IncorrectValidation> incorrectValidations = new ArrayList<>();
        for (Class<?> parentClazz : classes) {
            Field[] parentClassFields = parentClazz.getDeclaredFields();
            List<String> classesWithLackOfValidation = findClassesWithLackOfValidation(parentClassFields);

            if (!classesWithLackOfValidation.isEmpty()) {
                IncorrectValidation incorrectValidation = new IncorrectValidation(parentClazz.getCanonicalName(), classesWithLackOfValidation);
                incorrectValidations.add(incorrectValidation);
            }
        }
        return incorrectValidations;
    }

    private List<String> findClassesWithLackOfValidation(Field[] parentClassFields) {
        List<String> classesWithLackOfValidation = new ArrayList<>();

        for (Field parentClassField : parentClassFields) {
            boolean hasChildAnyValidationAnnotation = validationAnnotationChecker.hasAnyChildValidationAnnotation(parentClassField);
            if (hasChildAnyValidationAnnotation) {
                boolean isAnnotatedProperly = referenceValidator.isAnnotationPutOnReference(parentClassField);
                if (!isAnnotatedProperly) {
                    classesWithLackOfValidation.add(parentClassField.getName());
                }
            }
        }
        return classesWithLackOfValidation;
    }
}
