/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.ssl;

import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.Principal;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLEngineResult;
import javax.net.ssl.SSLException;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Reconfigurable;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.SslConfigs;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.security.ssl.SslEngineBuilder;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslFactory
implements Reconfigurable {
    private static final Logger log = LoggerFactory.getLogger(SslFactory.class);
    private final Mode mode;
    private final String clientAuthConfigOverride;
    private final boolean keystoreVerifiableUsingTruststore;
    private String endpointIdentification;
    private SslEngineBuilder sslEngineBuilder;

    public SslFactory(Mode mode) {
        this(mode, null, false);
    }

    public SslFactory(Mode mode, String clientAuthConfigOverride, boolean keystoreVerifiableUsingTruststore) {
        this.mode = mode;
        this.clientAuthConfigOverride = clientAuthConfigOverride;
        this.keystoreVerifiableUsingTruststore = keystoreVerifiableUsingTruststore;
    }

    @Override
    public void configure(Map<String, ?> configs) throws KafkaException {
        if (this.sslEngineBuilder != null) {
            throw new IllegalStateException("SslFactory was already configured.");
        }
        this.endpointIdentification = (String)configs.get("ssl.endpoint.identification.algorithm");
        HashMap<String, String> nextConfigs = new HashMap<String, String>();
        SslFactory.copyMapEntries(nextConfigs, configs, SslConfigs.NON_RECONFIGURABLE_CONFIGS);
        SslFactory.copyMapEntries(nextConfigs, configs, SslConfigs.RECONFIGURABLE_CONFIGS);
        SslFactory.copyMapEntry(nextConfigs, configs, "security.providers");
        if (this.clientAuthConfigOverride != null) {
            nextConfigs.put("ssl.client.auth", this.clientAuthConfigOverride);
        }
        SslEngineBuilder builder = new SslEngineBuilder(nextConfigs);
        if (this.keystoreVerifiableUsingTruststore) {
            try {
                SslEngineValidator.validate(builder, builder);
            }
            catch (Exception e) {
                throw new ConfigException("A client SSLEngine created with the provided settings can't connect to a server SSLEngine created with those settings.", (Object)e);
            }
        }
        this.sslEngineBuilder = builder;
    }

    @Override
    public Set<String> reconfigurableConfigs() {
        return SslConfigs.RECONFIGURABLE_CONFIGS;
    }

    @Override
    public void validateReconfiguration(Map<String, ?> newConfigs) {
        this.createNewSslEngineBuilder(newConfigs);
    }

    @Override
    public void reconfigure(Map<String, ?> newConfigs) throws KafkaException {
        SslEngineBuilder newSslEngineBuilder = this.createNewSslEngineBuilder(newConfigs);
        if (newSslEngineBuilder != this.sslEngineBuilder) {
            this.sslEngineBuilder = newSslEngineBuilder;
            log.info("Created new {} SSL engine builder with keystore {} truststore {}", new Object[]{this.mode, newSslEngineBuilder.keystore(), newSslEngineBuilder.truststore()});
        }
    }

    private SslEngineBuilder createNewSslEngineBuilder(Map<String, ?> newConfigs) {
        if (this.sslEngineBuilder == null) {
            throw new IllegalStateException("SslFactory has not been configured.");
        }
        HashMap<String, Object> nextConfigs = new HashMap<String, Object>(this.sslEngineBuilder.configs());
        SslFactory.copyMapEntries(nextConfigs, newConfigs, SslConfigs.RECONFIGURABLE_CONFIGS);
        if (this.clientAuthConfigOverride != null) {
            nextConfigs.put("ssl.client.auth", this.clientAuthConfigOverride);
        }
        if (!this.sslEngineBuilder.shouldBeRebuilt(nextConfigs)) {
            return this.sslEngineBuilder;
        }
        try {
            SslEngineBuilder newSslEngineBuilder = new SslEngineBuilder(nextConfigs);
            if (this.sslEngineBuilder.keystore() == null) {
                if (newSslEngineBuilder.keystore() != null) {
                    throw new ConfigException("Cannot add SSL keystore to an existing listener for which no keystore was configured.");
                }
            } else {
                if (newSslEngineBuilder.keystore() == null) {
                    throw new ConfigException("Cannot remove the SSL keystore from an existing listener for which a keystore was configured.");
                }
                if (!CertificateEntries.create(this.sslEngineBuilder.keystore().load()).equals(CertificateEntries.create(newSslEngineBuilder.keystore().load()))) {
                    throw new ConfigException("Keystore DistinguishedName or SubjectAltNames do not match");
                }
            }
            if (this.sslEngineBuilder.truststore() == null && newSslEngineBuilder.truststore() != null) {
                throw new ConfigException("Cannot add SSL truststore to an existing listener for which no truststore was configured.");
            }
            if (this.keystoreVerifiableUsingTruststore && (this.sslEngineBuilder.truststore() != null || this.sslEngineBuilder.keystore() != null)) {
                SslEngineValidator.validate(this.sslEngineBuilder, newSslEngineBuilder);
            }
            return newSslEngineBuilder;
        }
        catch (Exception e) {
            log.debug("Validation of dynamic config update of SSLFactory failed.", e);
            throw new ConfigException("Validation of dynamic config update of SSLFactory failed: " + e);
        }
    }

    public SSLEngine createSslEngine(String peerHost, int peerPort) {
        if (this.sslEngineBuilder == null) {
            throw new IllegalStateException("SslFactory has not been configured.");
        }
        return this.sslEngineBuilder.createSslEngine(this.mode, peerHost, peerPort, this.endpointIdentification);
    }

    @Deprecated
    public SSLContext sslContext() {
        return this.sslEngineBuilder.sslContext();
    }

    public SslEngineBuilder sslEngineBuilder() {
        return this.sslEngineBuilder;
    }

    private static <K, V> void copyMapEntries(Map<K, V> destMap, Map<K, ? extends V> srcMap, Set<K> keySet) {
        for (K k : keySet) {
            SslFactory.copyMapEntry(destMap, srcMap, k);
        }
    }

    private static <K, V> void copyMapEntry(Map<K, V> destMap, Map<K, ? extends V> srcMap, K key) {
        if (srcMap.containsKey(key)) {
            destMap.put(key, srcMap.get(key));
        }
    }

    private static class SslEngineValidator {
        private static final ByteBuffer EMPTY_BUF = ByteBuffer.allocate(0);
        private final SSLEngine sslEngine;
        private SSLEngineResult handshakeResult;
        private ByteBuffer appBuffer;
        private ByteBuffer netBuffer;

        static void validate(SslEngineBuilder oldEngineBuilder, SslEngineBuilder newEngineBuilder) throws SSLException {
            SslEngineValidator.validate(SslEngineValidator.createSslEngineForValidation(oldEngineBuilder, Mode.SERVER), SslEngineValidator.createSslEngineForValidation(newEngineBuilder, Mode.CLIENT));
            SslEngineValidator.validate(SslEngineValidator.createSslEngineForValidation(newEngineBuilder, Mode.SERVER), SslEngineValidator.createSslEngineForValidation(oldEngineBuilder, Mode.CLIENT));
        }

        private static SSLEngine createSslEngineForValidation(SslEngineBuilder sslEngineBuilder, Mode mode) {
            return sslEngineBuilder.createSslEngine(mode, "", 0, "");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        static void validate(SSLEngine clientEngine, SSLEngine serverEngine) throws SSLException {
            SslEngineValidator clientValidator = new SslEngineValidator(clientEngine);
            SslEngineValidator serverValidator = new SslEngineValidator(serverEngine);
            try {
                clientValidator.beginHandshake();
                serverValidator.beginHandshake();
                while (!serverValidator.complete() || !clientValidator.complete()) {
                    clientValidator.handshake(serverValidator);
                    serverValidator.handshake(clientValidator);
                }
            }
            finally {
                clientValidator.close();
                serverValidator.close();
            }
        }

        private SslEngineValidator(SSLEngine engine) {
            this.sslEngine = engine;
            this.appBuffer = ByteBuffer.allocate(this.sslEngine.getSession().getApplicationBufferSize());
            this.netBuffer = ByteBuffer.allocate(this.sslEngine.getSession().getPacketBufferSize());
        }

        void beginHandshake() throws SSLException {
            this.sslEngine.beginHandshake();
        }

        void handshake(SslEngineValidator peerValidator) throws SSLException {
            SSLEngineResult.HandshakeStatus handshakeStatus = this.sslEngine.getHandshakeStatus();
            block16: while (true) {
                switch (handshakeStatus) {
                    case NEED_WRAP: {
                        this.handshakeResult = this.sslEngine.wrap(EMPTY_BUF, this.netBuffer);
                        switch (this.handshakeResult.getStatus()) {
                            case OK: {
                                break;
                            }
                            case BUFFER_OVERFLOW: {
                                this.netBuffer.compact();
                                this.netBuffer = Utils.ensureCapacity(this.netBuffer, this.sslEngine.getSession().getPacketBufferSize());
                                this.netBuffer.flip();
                                break;
                            }
                            default: {
                                throw new SSLException("Unexpected handshake status: " + (Object)((Object)this.handshakeResult.getStatus()));
                            }
                        }
                        return;
                    }
                    case NEED_UNWRAP: {
                        if (peerValidator.netBuffer.position() == 0) {
                            return;
                        }
                        peerValidator.netBuffer.flip();
                        this.handshakeResult = this.sslEngine.unwrap(peerValidator.netBuffer, this.appBuffer);
                        peerValidator.netBuffer.compact();
                        handshakeStatus = this.handshakeResult.getHandshakeStatus();
                        switch (this.handshakeResult.getStatus()) {
                            case OK: {
                                continue block16;
                            }
                            case BUFFER_OVERFLOW: {
                                this.appBuffer = Utils.ensureCapacity(this.appBuffer, this.sslEngine.getSession().getApplicationBufferSize());
                                continue block16;
                            }
                            case BUFFER_UNDERFLOW: {
                                this.netBuffer = Utils.ensureCapacity(this.netBuffer, this.sslEngine.getSession().getPacketBufferSize());
                                continue block16;
                            }
                        }
                        throw new SSLException("Unexpected handshake status: " + (Object)((Object)this.handshakeResult.getStatus()));
                    }
                    case NEED_TASK: {
                        this.sslEngine.getDelegatedTask().run();
                        handshakeStatus = this.sslEngine.getHandshakeStatus();
                        continue block16;
                    }
                    case FINISHED: {
                        return;
                    }
                    case NOT_HANDSHAKING: {
                        if (this.handshakeResult.getHandshakeStatus() != SSLEngineResult.HandshakeStatus.FINISHED) {
                            throw new SSLException("Did not finish handshake");
                        }
                        return;
                    }
                }
                break;
            }
            throw new IllegalStateException("Unexpected handshake status " + (Object)((Object)handshakeStatus));
        }

        boolean complete() {
            return this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.FINISHED || this.sslEngine.getHandshakeStatus() == SSLEngineResult.HandshakeStatus.NOT_HANDSHAKING;
        }

        void close() {
            this.sslEngine.closeOutbound();
            try {
                this.sslEngine.closeInbound();
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
    }

    static class CertificateEntries {
        private final Principal subjectPrincipal;
        private final Set<List<?>> subjectAltNames;

        static List<CertificateEntries> create(KeyStore keystore) throws GeneralSecurityException {
            Enumeration<String> aliases = keystore.aliases();
            ArrayList<CertificateEntries> entries = new ArrayList<CertificateEntries>();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                Certificate cert = keystore.getCertificate(alias);
                if (!(cert instanceof X509Certificate)) continue;
                entries.add(new CertificateEntries((X509Certificate)cert));
            }
            return entries;
        }

        CertificateEntries(X509Certificate cert) throws GeneralSecurityException {
            this.subjectPrincipal = cert.getSubjectX500Principal();
            Collection<List<?>> altNames = cert.getSubjectAlternativeNames();
            this.subjectAltNames = altNames != null ? new HashSet(altNames) : Collections.emptySet();
        }

        public int hashCode() {
            return Objects.hash(this.subjectPrincipal, this.subjectAltNames);
        }

        public boolean equals(Object obj2) {
            if (!(obj2 instanceof CertificateEntries)) {
                return false;
            }
            CertificateEntries other = (CertificateEntries)obj2;
            return Objects.equals(this.subjectPrincipal, other.subjectPrincipal) && Objects.equals(this.subjectAltNames, other.subjectAltNames);
        }

        public String toString() {
            return "subjectPrincipal=" + this.subjectPrincipal + ", subjectAltNames=" + this.subjectAltNames;
        }
    }
}

