package plus.easydo.starter.plugins.gen.service.impl;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.apache.velocity.runtime.resource.loader.StringResourceLoader;
import org.apache.velocity.runtime.resource.util.StringResourceRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.RowMapperResultSetExtractor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import plus.easydo.core.exception.BaseException;
import plus.easydo.jdbc.utils.DataSourceExecTool;
import plus.easydo.starter.plugins.gen.constant.GenConstants;
import plus.easydo.starter.plugins.gen.entity.GenTable;
import plus.easydo.starter.plugins.gen.entity.GenTableColumn;
import plus.easydo.starter.plugins.gen.rowmapper.GenTableColumnRowMapper;
import plus.easydo.starter.plugins.gen.rowmapper.GenTableRowMapper;
import plus.easydo.starter.plugins.gen.mapper.GenTableColumnMapper;
import plus.easydo.starter.plugins.gen.mapper.GenTableMapper;
import plus.easydo.starter.plugins.gen.service.IGenTableService;
import plus.easydo.starter.plugins.gen.service.TemplateManagementService;
import plus.easydo.starter.plugins.gen.util.GenUtils;
import plus.easydo.starter.plugins.gen.util.VelocityInitializer;
import plus.easydo.starter.plugins.gen.util.VelocityUtils;
import plus.easydo.core.constant.Constants;
import plus.easydo.core.exception.CustomException;
import plus.easydo.starter.plugins.gen.vo.TemplateManagementVo;
import plus.easydo.utils.StringUtils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 业务 服务层实现
 *
 * @author laoyu
 */
@Service
public class GenTableServiceImpl extends ServiceImpl<GenTableMapper, GenTable> implements IGenTableService {

    private static final Logger log = LoggerFactory.getLogger(GenTableServiceImpl.class);

    private static final String vm = ".vm";

    private static final RowMapperResultSetExtractor<GenTable> genTableSetExtractor = new RowMapperResultSetExtractor<>(new GenTableRowMapper());

    private static final RowMapperResultSetExtractor<GenTableColumn> genTableColumnSetExtractor = new RowMapperResultSetExtractor<>(new GenTableColumnRowMapper());

    @Autowired
    private GenTableColumnMapper genTableColumnMapper;

    @Autowired
    private TemplateManagementService templateManagementService;

    @Autowired
    private DataSourceExecTool dataSourceExecTool;

    @Autowired
    VelocityInitializer velocityInitializer;

    /**
     * 查询业务信息
     *
     * @param id 业务ID
     * @return 业务信息
     */
    @Override
    public GenTable selectGenTableById(Long id) {
        GenTable genTable = baseMapper.selectGenTableById(id);
        setTableFromOptions(genTable);
        return genTable;
    }

    /**
     * 查询业务列表
     *
     * @param genTable 业务信息
     * @return 业务集合
     */
    @Override
    public List<GenTable> selectGenTableList(GenTable genTable) {
        return baseMapper.selectGenTableList(genTable);
    }

    /**
     * 查询据库列表
     *
     * @param genTable 业务信息
     * @return 数据库表集合
     */
    @Override
    public List<GenTable> selectDbTableList(GenTable genTable) {
        StringBuilder stringBuilder = new StringBuilder();
        String sql = "select table_name, table_comment, create_time, update_time from information_schema.tables where table_schema = (select database()) AND table_name NOT LIKE 'qrtz_%' AND table_name NOT LIKE 'gen_%'"; // AND table_name NOT IN (select table_name from gen_table) ";
        stringBuilder.append(sql);
        if (StrUtil.isNotBlank(genTable.getTableName())){
            stringBuilder.append(" AND lower(table_name) like lower(concat('%', ").append("'").append(genTable.getTableName()).append("'").append(", '%')) ");
        }
        if (StrUtil.isNotBlank(genTable.getTableComment())){
            stringBuilder.append(" AND lower(table_comment) like lower(concat('%', ").append("'").append(genTable.getTableComment()).append("'").append(", '%')) ");
        }
        if (StrUtil.isNotBlank(genTable.getTableName())){
            stringBuilder.append("AND lower(table_name) like lower(concat('%', ").append("'").append(genTable.getTableName()).append("'").append(", '%'))");
        }
        return dataSourceExecTool.query(genTable.getDataSourceId(), stringBuilder.toString(), genTableSetExtractor);
    }

    /**
     * 查询据库列表
     *
     *
     * @param dataSourceId 数据源
     * @param tableNames 表名称组
     * @return 数据库表集合
     */
    @Override
    public List<GenTable> selectDbTableListByNames(String dataSourceId, String[] tableNames) {
        StringBuilder startBuilder = new StringBuilder();
        StringBuilder endBuilder = new StringBuilder();
        String sql = "select table_name, table_comment, create_time, update_time from information_schema.tables where table_name NOT LIKE 'qrtz_%' and table_name NOT LIKE 'gen_%' and table_schema = (select database()) ";
        startBuilder.append(sql);
        endBuilder.append(" and table_name in (");
        for (String tableName : tableNames) {
            endBuilder.append("'").append(tableName).append("'").append(",");
        }
        String endSql = endBuilder.substring(0,endBuilder.length()-1);
        startBuilder.append(endSql).append(")");
        return dataSourceExecTool.query(dataSourceId,startBuilder.toString(), genTableSetExtractor);
    }

    /**
     * 查询所有表信息
     *
     * @return 表信息集合
     */
    @Override
    public List<GenTable> selectGenTableAll() {
        return baseMapper.selectGenTableAll();
    }

    /**
     * 修改业务
     *
     * @param genTable 业务信息
     */
    @Override
    @Transactional
    public void updateGenTable(GenTable genTable) {
        String options = JSON.toJSONString(genTable.getParams());
        genTable.setOptions(options);
        int row = baseMapper.updateGenTable(genTable);
        if (row > 0) {
            for (GenTableColumn cenTableColumn : genTable.getColumns()) {
                genTableColumnMapper.updateGenTableColumn(cenTableColumn);
            }
        }
    }

    /**
     * 删除业务对象
     *
     * @param tableIds 需要删除的数据ID
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void deleteGenTableByIds(Long[] tableIds) {
        baseMapper.deleteGenTableByIds(tableIds);
        genTableColumnMapper.deleteGenTableColumnByIds(tableIds);
    }

    /**
     * 导入表结构
     *
     * @param dataSourceId 数据源
     * @param tableList 导入表列表
     */
    @Override
    @Transactional(rollbackFor = {Exception.class})
    public void importGenTable(String dataSourceId, List<GenTable> tableList) {
        String operationName = "easy-do";
        try {
            for (GenTable table : tableList) {
                String tableName = table.getTableName();
                GenUtils.initTable(table, operationName);
                int row = baseMapper.insertGenTable(table);
                if (row > 0) {
                    // 保存列信息
                    List<GenTableColumn> genTableColumns = selectDbTableColumnsByName(dataSourceId,tableName);
                    for (GenTableColumn column : genTableColumns) {
                        GenUtils.initColumnField(column, table);
                        genTableColumnMapper.insertGenTableColumn(column);
                    }
                }
            }
        } catch (Exception e) {
            throw new CustomException("导入失败：" + e.getMessage());
        }
    }

    /**
     * 根据数据源id，和表明 动态获取表的所有字段列信息
     *
     * @param dataSourceId dataSourceId
     * @param tableName tableName
     * @return java.util.List
     * @author laoyu
     */
    private List<GenTableColumn> selectDbTableColumnsByName(String dataSourceId, String tableName){
        String sql = "select column_name, (case when (is_nullable = 'no' && column_key != 'PRI') then '1' else null end) as is_required, (case when column_key = 'PRI' then '1' else '0' end) as is_pk, ordinal_position as sort, column_comment, (case when extra = 'auto_increment' then '1' else '0' end) as is_increment, column_type " +
                " from information_schema.columns where table_schema = (select database()) and table_name = '" + tableName + "'" + " order by ordinal_position";
        return dataSourceExecTool.query(dataSourceId, sql ,genTableColumnSetExtractor);
    }

    /**
     * 预览代码
     *
     * @param tableId 表编号
     * @return 预览数据列表
     */
    @Override
    public Map<String, String> previewCode(Long tableId) {
        Map<String, String> dataMap = new LinkedHashMap<>();
        // 查询表信息
        GenTable table = baseMapper.selectGenTableById(tableId);
        // 设置主子表信息
        setSubTable(table);
        // 设置主键列信息
        setPkColumn(table);
        VelocityContext context = VelocityUtils.prepareContext(table);
        String templateIds = table.getTemplateIds();
        Map<TemplateManagementVo, Template> templates = getTemplatesForDb(templateIds);
        for ( Map.Entry<TemplateManagementVo, Template> entry: templates.entrySet()){
            StringWriter stringWriter = new StringWriter();
            TemplateManagementVo vo = entry.getKey();
            Template template = entry.getValue();
            template.merge(context, stringWriter);
            dataMap.put(vo.getFileName(), stringWriter.toString());
        }
        return dataMap;
    }

    /**
     * 生成代码（下载方式）
     *
     * @param tableName 表名称
     * @return 数据
     */
    @Override
    public byte[] downloadCode(String tableName) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ZipOutputStream zip = new ZipOutputStream(outputStream);
        generatorCode(tableName, zip);
        IOUtils.closeQuietly(zip);
        return outputStream.toByteArray();
    }

    /**
     * 生成代码（自定义路径）
     *
     * @param tableName 表名称
     */
    @Override
    public void generatorCode(String tableName) {
        // 查询表信息
        GenTable table = baseMapper.selectGenTableByName(tableName);
        // 设置主子表信息
        setSubTable(table);
        // 设置主键列信息
        setPkColumn(table);

        VelocityContext context = VelocityUtils.prepareContext(table);

        // 获取模板列表
        String templateIds = table.getTemplateIds();
        Map<TemplateManagementVo, Template> templates = getTemplatesForDb(templateIds);
        for ( Map.Entry<TemplateManagementVo, Template> entry: templates.entrySet()){
            StringWriter stringWriter = new StringWriter();
            TemplateManagementVo vo = entry.getKey();
            Template template = entry.getValue();
            template.merge(context, stringWriter);
            try {
                String path = VelocityUtils.generatePath(table,vo.getFilePath());
                FileUtils.writeStringToFile(new File(path), stringWriter.toString(), Constants.UTF8);
            } catch (IOException e) {
                throw new CustomException("渲染模板失败，表名：" + table.getTableName());
            }
        }

    }

    /**
     * 同步数据库
     *
     * @param tableName 表名称
     */
    @Override
    @Transactional
    public void synchDb(String tableName) {
        GenTable table = baseMapper.selectGenTableByName(tableName);
        List<GenTableColumn> tableColumns = table.getColumns();
        List<String> tableColumnNames = tableColumns.stream().map(GenTableColumn::getColumnName).collect(Collectors.toList());

        List<GenTableColumn> dbTableColumns = genTableColumnMapper.selectDbTableColumnsByName(tableName);
        if (StringUtils.isEmpty(dbTableColumns)) {
            throw new CustomException("同步数据失败，原表结构不存在");
        }
        List<String> dbTableColumnNames = dbTableColumns.stream().map(GenTableColumn::getColumnName).collect(Collectors.toList());

        dbTableColumns.forEach(column -> {
            if (!tableColumnNames.contains(column.getColumnName())) {
                GenUtils.initColumnField(column, table);
                genTableColumnMapper.insertGenTableColumn(column);
            }
        });

        List<GenTableColumn> delColumns = tableColumns.stream().filter(column -> !dbTableColumnNames.contains(column.getColumnName())).collect(Collectors.toList());
        if (StringUtils.isNotEmpty(delColumns)) {
            genTableColumnMapper.deleteGenTableColumns(delColumns);
        }
    }

    /**
     * 批量生成代码（下载方式）
     *
     * @param tableNames 表数组
     * @return 数据
     */
    @Override
    public byte[] downloadCode(String[] tableNames) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ZipOutputStream zip = new ZipOutputStream(outputStream);
        for (String tableName : tableNames) {
            generatorCode(tableName, zip);
        }
        IOUtils.closeQuietly(zip);
        return outputStream.toByteArray();
    }

    /**
     * 查询表信息并生成代码
     *
     * @param tableName tableName
     * @param zip zip
     * @author laoyu
     */
    private void generatorCode(String tableName, ZipOutputStream zip) {
        // 查询表信息
        GenTable table = baseMapper.selectGenTableByName(tableName);

        // 设置主子表信息
        setSubTable(table);
        // 设置主键列信息
        setPkColumn(table);

        VelocityContext context = VelocityUtils.prepareContext(table);

        // 获取模板列表
        String templateIds = table.getTemplateIds();
        Map<TemplateManagementVo, Template> templates = getTemplatesForDb(templateIds);
        for ( Map.Entry<TemplateManagementVo, Template> entry: templates.entrySet()){
            StringWriter stringWriter = new StringWriter();
            TemplateManagementVo vo = entry.getKey();
            Template template = entry.getValue();
            template.merge(context, stringWriter);
            try {
                // 添加到zip
                zip.putNextEntry(new ZipEntry(VelocityUtils.generatePath(table,vo.getFilePath())));
                IOUtils.write(stringWriter.toString(), zip, Constants.UTF8);
                IOUtils.closeQuietly(stringWriter);
                zip.flush();
                zip.closeEntry();
            } catch (IOException e) {
                log.error("渲染模板失败，表名：" + table.getTableName(), e);
            }
        }

    }

    /**
     * 修改保存参数校验
     *
     * @param genTable 业务信息
     */
    @Override
    public void validateEdit(GenTable genTable) {
        if (GenConstants.TPL_TREE.equals(genTable.getTplCategory())) {
            String options = JSON.toJSONString(genTable.getParams());
            JSONObject paramsObj = JSONObject.parseObject(options);
            if (StringUtils.isEmpty(paramsObj.getString(GenConstants.TREE_CODE))) {
                throw new CustomException("树编码字段不能为空");
            } else if (StringUtils.isEmpty(paramsObj.getString(GenConstants.TREE_PARENT_CODE))) {
                throw new CustomException("树父编码字段不能为空");
            } else if (StringUtils.isEmpty(paramsObj.getString(GenConstants.TREE_NAME))) {
                throw new CustomException("树名称字段不能为空");
            } else if (GenConstants.TPL_SUB.equals(genTable.getTplCategory())) {
                if (StringUtils.isEmpty(genTable.getSubTableName())) {
                    throw new CustomException("关联子表的表名不能为空");
                } else if (StringUtils.isEmpty(genTable.getSubTableFkName())) {
                    throw new CustomException("子表关联的外键名不能为空");
                }
            }
        }
    }

    /**
     * 设置主键列信息
     *
     * @param table 业务表信息
     */
    public void setPkColumn(GenTable table) {
        for (GenTableColumn column : table.getColumns()) {
            if (column.isPk()) {
                table.setPkColumn(column);
                break;
            }
        }
        if (StringUtils.isNull(table.getPkColumn())) {
            table.setPkColumn(table.getColumns().get(0));
        }
        if (GenConstants.TPL_SUB.equals(table.getTplCategory())) {
            for (GenTableColumn column : table.getSubTable().getColumns()) {
                if (column.isPk()) {
                    table.getSubTable().setPkColumn(column);
                    break;
                }
            }
            if (StringUtils.isNull(table.getSubTable().getPkColumn())) {
                table.getSubTable().setPkColumn(table.getSubTable().getColumns().get(0));
            }
        }
    }

    /**
     * 设置主子表信息
     *
     * @param table 业务表信息
     */
    public void setSubTable(GenTable table) {
        String subTableName = table.getSubTableName();
        if (StringUtils.isNotEmpty(subTableName)) {
            table.setSubTable(baseMapper.selectGenTableByName(subTableName));
        }
    }

    /**
     * 设置代码生成其他选项值
     *
     * @param genTable 设置后的生成对象
     */
    public void setTableFromOptions(GenTable genTable) {
        JSONObject paramsObj = JSONObject.parseObject(genTable.getOptions());
        if (StringUtils.isNotNull(paramsObj)) {
            String treeCode = paramsObj.getString(GenConstants.TREE_CODE);
            String treeParentCode = paramsObj.getString(GenConstants.TREE_PARENT_CODE);
            String treeName = paramsObj.getString(GenConstants.TREE_NAME);
            String parentMenuId = paramsObj.getString(GenConstants.PARENT_MENU_ID);
            String parentMenuName = paramsObj.getString(GenConstants.PARENT_MENU_NAME);
            genTable.setTreeCode(treeCode);
            genTable.setTreeParentCode(treeParentCode);
            genTable.setTreeName(treeName);
            genTable.setParentMenuId(parentMenuId);
            genTable.setParentMenuName(parentMenuName);
        }
    }

    /**
     * 获取代码生成地址
     *
     * @param table    业务表信息
     * @param template 模板文件路径
     * @return 生成地址
     */
    @Deprecated
    public static String getGenPath(GenTable table, String template) {
        String genPath = table.getGenPath();
        if (StringUtils.equals(genPath, "/")) {
            return System.getProperty("user.dir") + File.separator + "src" + File.separator + VelocityUtils.getFileName(template, table);
        }
        return genPath + File.separator + VelocityUtils.getFileName(template, table);
    }


    /**
     * 从本地文件获取模板
     *
     * @param genTable genTable
     * @return java.util.List
     * @author laoyu
     */
    Map<String,Template> getTemplatesForLocalhost(GenTable genTable){
        Map<String,Template> templates = new HashMap<>();
        // 获取本地模板列表
        VelocityInitializer.initVelocityClasspathResourceLoader();
        List<String> filePaths = VelocityUtils.getTemplatePathList(genTable.getIsManager(),genTable.getTplCategory());
        for (String filePath : filePaths) {
            // 渲染模板
            templates.put(filePath,Velocity.getTemplate(filePath, Constants.UTF8));
        }
        return templates;
    }

    /**
     * 从数据库获取模板
     *
     * @param templateIds templateIds
     * @return java.util.List
     * @author laoyu
     */
    Map<TemplateManagementVo,Template> getTemplatesForDb(String templateIds){
        VelocityInitializer.initVelocityStringResourceLoader();
        Map<TemplateManagementVo,Template> templates = new HashMap<>();
        if (StrUtil.isEmpty(templateIds)){
            throw new BaseException("未绑定模板");
        }
        String[] ids = templateIds.split(",");
        List<TemplateManagementVo> templateManagementVos = templateManagementService.selectByIds(ids);
        //临时文件夹
        for (TemplateManagementVo vo:templateManagementVos){
            String code = vo.getTemplateCode();
            if(StrUtil.isNotBlank(code)){
                try{
                    //生成临时模板文件
                    String fileName = vo.getFileName();
                    StringResourceRepository repo = StringResourceLoader.getRepository();
                    repo.putStringResource(fileName, vo.getTemplateCode());
                    Template template = Velocity.getTemplate(fileName);
                    templates.put(vo,template);
                }catch (Exception e){
                    throw new BaseException("从数据库读取模板文件异常：",e.getMessage());
                }
            }
        }
        return templates;
    }
}
