package plus.gaga.middleware.sdk.infrastructure.git;

import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import plus.gaga.middleware.sdk.types.utils.RandomStringUtils;

import java.io.*;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * @author Fuzhengwei bugstack.cn @小傅哥
 * @description Git 命令
 * @create 2024-07-02 08:05
 */
public class GitCommand {

    private final Logger logger = LoggerFactory.getLogger(GitCommand.class);

    private final String githubReviewLogUri;
    private final String githubToken;
    private final String project;
    private final String branch;
    private final String author;
    private final String message;

    public GitCommand(String githubReviewLogUri, String githubToken, String project, String branch, String author, String message) {
        this.githubReviewLogUri = githubReviewLogUri;
        this.githubToken = githubToken;
        this.project = project;
        this.branch = branch;
        this.author = author;
        this.message = message;
    }

    /**
     * 获取代码提交差异
     */
    public String diff() throws IOException, InterruptedException {
        ProcessBuilder processBuilder = new ProcessBuilder("git", "diff", "HEAD~1", "HEAD");
        processBuilder.directory(new java.io.File("."));

        Process process = processBuilder.start();

        StringBuilder diffCode = new StringBuilder();
        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
        String line;
        while ((line = reader.readLine()) != null) {
            diffCode.append(line);
        }

        int exitCode = process.waitFor();
        logger.info("openai-code-review git diff done! {}", exitCode);

        return diffCode.toString();
    }

    /**
     * 提交推送代码
     *
     * @param recommend 代码评审记录
     * @return 日志地址
     */
    public String commitAndPush(String recommend) {
        try {
            // 克隆仓库
            Git git = Git.cloneRepository()
                    .setURI(githubReviewLogUri + ".git")
                    .setDirectory(new File("repo"))
                    .setCredentialsProvider(new UsernamePasswordCredentialsProvider(githubToken, ""))
                    .call();

            // 创建分支
            String dateFolderName = new SimpleDateFormat("yyyy-MM-dd").format(new Date());
            File dateFolder = new File("repo/" + dateFolderName);
            if (!dateFolder.exists()) {
                dateFolder.mkdirs();
            }

            // 文件名称；工程名称、分支名称、时间戳、随机数(4位)
            String fileName = project + "-" + branch + "_" + author + "_" + System.currentTimeMillis() + "_" + RandomStringUtils.randomNumeric(4) + ".md";
            File newFile = new File(dateFolder, fileName);
            try (FileWriter writer = new FileWriter(newFile)) {
                writer.write(recommend);
            }

            // 提交内容
            git.add().addFilepattern(dateFolderName + "/" + fileName).call();
            git.commit().setMessage("add code review new file" + fileName).call();
            git.push().setCredentialsProvider(new UsernamePasswordCredentialsProvider(githubToken, "")).call();

            logger.info("openai-code-review git commit and push done! {}", fileName);

            return githubReviewLogUri + "/blob/master/" + dateFolderName + "/" + fileName;
        } catch (GitAPIException | IOException e) {
            e.printStackTrace();
            return "";
        }
    }

    public String getGithubReviewLogUri() {
        return githubReviewLogUri;
    }

    public String getGithubToken() {
        return githubToken;
    }

    public String getProject() {
        return project;
    }

    public String getBranch() {
        return branch;
    }

    public String getAuthor() {
        return author;
    }

    public String getMessage() {
        return message;
    }
}
