/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.base;

import io.swagger.v3.oas.annotations.Operation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.api.Result;
import plus.hiver.common.utils.PageUtil;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.vo.PageVo;

import java.io.Serializable;
import java.util.List;

/**
 * 控制器基类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
public abstract class HiverBaseController<E, ID extends Serializable> {
    @Autowired
    public abstract HiverBaseService<E, ID> getService();

    @GetMapping(value = "/get/{id}")
    @ResponseBody
    @Operation(summary = "通过id获取")
    public Result<E> get(@PathVariable ID id) {
        E entity = getService().get(id);
        return new ResultUtil<E>().setData(entity);
    }

    @GetMapping(value = "/getAll")
    @ResponseBody
    @Operation(summary = "获取全部数据")
    public Result<List<E>> getAll() {
        List<E> list = getService().getAll();
        return new ResultUtil<List<E>>().setData(list);
    }

    @GetMapping(value = "/getByPage")
    @ResponseBody
    @Operation(summary = "分页获取")
    public Result<Page<E>> getByPage(PageVo page) {
        Page<E> data = getService().findAll(PageUtil.initPage(page));
        return new ResultUtil<Page<E>>().setData(data);
    }

    @PostMapping(value = "/save")
    @ResponseBody
    @Operation(summary = "保存数据")
    public Result<E> save(E entity) {
        E e = getService().save(entity);
        return new ResultUtil<E>().setData(e);
    }

    @PutMapping(value = "/update")
    @ResponseBody
    @Operation(summary = "更新数据")
    public Result<E> update(E entity) {
        E e = getService().update(entity);
        return new ResultUtil<E>().setData(e);
    }

    @PostMapping(value = "/delByIds")
    @ResponseBody
    @Operation(summary = "批量通过id删除")
    public Result<Object> delByIds(ID[] ids) {
        for (ID id : ids) {
            getService().delete(id);
        }
        return ResultUtil.success("批量通过id删除数据成功");
    }
}
