/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.base;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;

import java.io.Serializable;
import java.util.List;

/**
 * 数据服务基类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@FunctionalInterface
public interface HiverBaseService<E, ID extends Serializable> {
    HiverBaseDao<E, ID> getRepository();

    /**
     * 根据ID获取 不存在且使用返回的对象时会抛异常
     *
     * @param id 索引
     * @return 实体
     */
    default E get(ID id) {
        return getRepository().getReferenceById(id);
    }

    /**
     * 根据ID获取 不存在则返回null
     *
     * @param id 索引
     * @return 实体
     */
    default E findById(ID id) {
        return getRepository().findById(id).orElse(null);
    }

    /**
     * 获取所有列表
     *
     * @return 实体列表
     */
    default List<E> getAll() {
        return getRepository().findAll();
    }

    /**
     * 获取总数
     *
     * @return 实体总数
     */
    default Long getTotalCount() {
        return getRepository().count();
    }

    /**
     * 保存
     *
     * @param entity 实体
     * @return 实体
     */
    default E save(E entity) {
        return getRepository().save(entity);
    }

    /**
     * 修改
     *
     * @param entity 实体
     * @return 实体
     */
    default E update(E entity) {
        return getRepository().saveAndFlush(entity);
    }

    /**
     * 批量保存与修改
     *
     * @param entities 实体列表
     * @return 实体列表
     */
    default Iterable<E> saveOrUpdateAll(Iterable<E> entities) {
        return getRepository().saveAll(entities);
    }

    /**
     * 删除
     *
     * @param entity 实体
     */
    default void delete(E entity) {
        getRepository().delete(entity);
    }

    /**
     * 根据Id删除
     *
     * @param id 索引
     */
    default void delete(ID id) {
        getRepository().deleteById(id);
    }

    /**
     * 批量根据id删除
     *
     * @param ids 索引列表
     */
    default void deleteAllById(Iterable<ID> ids) {
        getRepository().deleteAllById(ids);
    }

    /**
     * 批量删除
     *
     * @param entities 实体列表
     */
    default void delete(Iterable<E> entities) {
        getRepository().deleteAll(entities);
    }

    /**
     * 清空缓存，提交持久化
     */
    default void flush() {
        getRepository().flush();
    }

    /**
     * 根据条件查询获取
     *
     * @param spec 查询条件
     * @return 实体列表
     */
    default List<E> findAll(Specification<E> spec) {
        return getRepository().findAll(spec);
    }

    /**
     * 分页获取
     *
     * @param pageable 分页条件
     * @return 实体列表
     */
    default Page<E> findAll(Pageable pageable) {
        return getRepository().findAll(pageable);
    }

    /**
     * 根据查询条件分页获取
     *
     * @param spec     查询条件
     * @param pageable 分页条件
     * @return 实体列表
     */
    default Page<E> findAll(Specification<E> spec, Pageable pageable) {
        return getRepository().findAll(spec, pageable);
    }

    /**
     * 获取查询条件的结果数
     *
     * @param spec 查询条件
     * @return 结果数
     */
    default long count(Specification<E> spec) {
        return getRepository().count(spec);
    }
}
