/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.interceptor;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;
import plus.hiver.common.annotation.RateLimiter;
import plus.hiver.common.config.properties.HiverIpLimitProperties;
import plus.hiver.common.config.properties.HiverLimitProperties;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.limit.RedisRaterLimiter;
import plus.hiver.common.service.SettingService;
import plus.hiver.common.utils.IpInfoUtil;
import plus.hiver.common.vo.OtherSetting;

import java.lang.reflect.Method;

/**
 * 限流拦截器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class LimitRaterInterceptor implements HandlerInterceptor {
    @Autowired
    private HiverLimitProperties limitProperties;

    @Autowired
    private HiverIpLimitProperties ipLimitProperties;

    @Autowired
    private RedisRaterLimiter redisRaterLimiter;

    @Autowired
    private IpInfoUtil ipInfoUtil;

    @Autowired
    private SettingService settingService;

    public OtherSetting getOtherSetting() {
        Setting setting = settingService.get(SettingConstant.OTHER_SETTING);
        if (StrUtil.isBlank(setting.getValue())) {
            return null;
        }
        return new Gson().fromJson(setting.getValue(), OtherSetting.class);
    }

    /**
     * 预处理回调方法，实现处理器的预处理（如登录检查）
     * 第三个参数为响应的处理器，即controller
     * 返回true，表示继续流程，调用下一个拦截器或者处理器
     * 返回false，表示流程中断，通过response产生响应
     */
    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response,
                             Object handler) throws Exception {
        String ip = ipInfoUtil.getIpAddr(request);

        if (ipLimitProperties.getEnable()) {
            Boolean token1 = redisRaterLimiter.acquireByRedis(ip,
                    ipLimitProperties.getLimit(), ipLimitProperties.getTimeout());
            if (!token1) {
                throw new HiverException("你手速怎么这么快，请点慢一点");
            }
        }

        if (limitProperties.getEnable()) {
            Boolean token2 = redisRaterLimiter.acquireByRedis(HiverConstant.LIMIT_ALL,
                    limitProperties.getLimit(), limitProperties.getTimeout());
            if (!token2) {
                throw new HiverException("当前访问总人数太多啦，请稍后再试");
            }
        }

        // IP黑名单
        OtherSetting os = getOtherSetting();
        if (os != null && StrUtil.isNotBlank(os.getBlacklist())) {
            String[] list = os.getBlacklist().split("\n");
            for (String item : list) {
                if (item.equals(ip)) {
                    throw new HiverException("您的IP已被添加至黑名单，请滚");
                }
            }
        }

        try {
            HandlerMethod handlerMethod = (HandlerMethod) handler;
            Object bean = handlerMethod.getBean();
            Method method = handlerMethod.getMethod();
            RateLimiter rateLimiter = method.getAnnotation(RateLimiter.class);
            if (rateLimiter != null) {
                String name = rateLimiter.name();
                Long limit = rateLimiter.rate();
                Long timeout = rateLimiter.rateInterval();
                if (StrUtil.isBlank(name)) {
                    name = StrUtil.subBefore(bean.toString(), "@", false) + "_" + method.getName();
                }
                if (rateLimiter.ipLimit()) {
                    name += "_" + ip;
                }
                Boolean token3 = redisRaterLimiter.acquireByRedis(name, limit, timeout);
                if (!token3) {
                    String msg = "当前访问人数太多啦，请稍后再试";
                    if (rateLimiter.ipLimit()) {
                        msg = "你手速怎么这么快，请点慢一点";
                    }
                    throw new HiverException(msg);
                }
            }
        } catch (HiverException e) {
            throw new HiverException(e.getMsg());
        } catch (Exception e) {

        }
        return true;
    }

    /**
     * 当前请求进行处理之后，也就是Controller方法调用之后执行，
     * 但是它会在DispatcherServlet 进行视图返回渲染之前被调用。
     * 此时我们可以通过modelAndView对模型数据进行处理或对视图进行处理。
     */
    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response,
                           Object handler, ModelAndView modelAndView) throws Exception {
    }

    /**
     * 方法将在整个请求结束之后，也就是在DispatcherServlet渲染了对应的视图之后执行。
     * 这个方法的主要作用是用于进行资源清理工作的。
     */
    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
                                Object handler, Exception ex) throws Exception {
    }
}
