/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.lock;

import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

/**
 * Redis锁模板
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class RedisLockTemplate implements DistributedLockTemplate {
    @Autowired
    private RedissonClient redisson;

    @Override
    public Object execute(String lockId, Integer timeout, Integer leaseTime, TimeUnit unit, Callback callback) {
        if (timeout == null) {
            timeout = 0;
        }
        RLock lock = null;
        boolean getLock = false;
        try {
            lock = redisson.getLock(lockId);
            if (leaseTime == null || leaseTime <= 0) {
                getLock = lock.tryLock(timeout, unit);
            } else {
                getLock = lock.tryLock(timeout, leaseTime, unit);
            }
            if (getLock) {
                // 拿到锁
                return callback.onGetLock();
            } else {
                // 未拿到锁
                return callback.onTimeout();
            }
        } catch (InterruptedException ex) {
            log.error(ex.getMessage(), ex);
            Thread.currentThread().interrupt();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        } finally {
            if (getLock) {
                // 释放锁
                lock.unlock();
            }
        }
        return null;
    }
}
