/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.serviceimpl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.dao.DepartmentDao;
import plus.hiver.common.entity.Department;
import plus.hiver.common.service.DepartmentService;
import plus.hiver.common.utils.SecurityUtil;

import java.util.ArrayList;
import java.util.List;

/**
 * 部门接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class DepartmentServiceImpl implements DepartmentService {
    @Autowired
    private DepartmentDao departmentDao;

    @Autowired
    private SecurityUtil securityUtil;

    @Override
    public DepartmentDao getRepository() {
        return departmentDao;
    }

    @Override
    public List<Department> findByParentIdOrderBySortOrder(Long parentId, Boolean openDataFilter) {
        // 数据权限
        List<Long> depIds = securityUtil.getDeparmentIds();
        if (depIds != null && depIds.size() > 0 && openDataFilter) {
            return departmentDao.findByParentIdAndIdInOrderBySortOrder(parentId, depIds);
        }
        return departmentDao.findByParentIdOrderBySortOrder(parentId);
    }

    @Override
    public List<Department> findByParentIdAndStatusOrderBySortOrder(Long parentId, Integer status) {
        return departmentDao.findByParentIdAndStatusOrderBySortOrder(parentId, status);
    }

    @Override
    public List<Department> findByTitleLikeOrderBySortOrder(String title, Boolean openDataFilter) {
        // 数据权限
        List<Long> depIds = securityUtil.getDeparmentIds();
        if (depIds != null && depIds.size() > 0 && openDataFilter) {
            return departmentDao.findByTitleLikeAndIdInOrderBySortOrder(title, depIds);
        }
        return departmentDao.findByTitleLikeOrderBySortOrder(title);
    }

    @Override
    public List<Department> findDepartmentAllById(Long parentId) {
        List<Department> departments = new ArrayList<>();
        if (!HiverConstant.PARENT_ID.equals(parentId)) {
            Department parent = departmentDao.getReferenceById(parentId);
            while (parent != null) {
                departments.add(parent);
                parent = getNextParentId(parent.getParentId());
            }
        }
        return departments;
    }

    @Override
    public List<Department> findByDeletedAndStatus(Integer delFlag, Integer status) {
        return departmentDao.findByDeletedAndStatus(delFlag, status);
    }

    @Override
    public List<Department> findByDeletedAndStatusAndParentId(Integer delFlag, Integer status, Long parentId) {
        return departmentDao.findByDeletedAndStatusAndParentId(delFlag, status, parentId);
    }

    private Department getNextParentId(Long id) {
        return departmentDao.getReferenceById(id);
    }
}
