/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.utils;

import cn.hutool.core.util.StrUtil;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.dao.DepartmentDao;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.entity.Department;
import plus.hiver.common.entity.Role;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.mybatis.IUserRoleService;
import plus.hiver.common.vo.TokenUser;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 安全工具类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Component
public class SecurityUtil {
    @Autowired
    private IUserRoleService iUserRoleService;

    @Autowired
    private DepartmentDao departmentDao;

    @Autowired
    private RedisTemplateHelper redisTemplate;

    /**
     * 获取当前用户数据权限 null代表具有所有权限 包含值为-1的数据代表无任何权限 包含值为-2的数据代表仅自己的权限
     */
    public List<Long> getDeparmentIds() {
        List<Long> deparmentIds = new ArrayList<>();
        User u = getCurrUserSimple();
        // 读取缓存
        String key = "userRole::depIds:" + u.getId();
        String v = redisTemplate.get(key);
        if (StrUtil.isNotBlank(v)) {
            deparmentIds = new Gson().fromJson(v, new TypeToken<List<String>>() {
            }.getType());
            return deparmentIds;
        }
        // 当前用户拥有角色
        List<Role> roles = iUserRoleService.findByUserId(u.getId());
        // 判断有无全部数据的角色
        Boolean flagAll = false;
        for (Role r : roles) {
            if (r.getDataType() == null || r.getDataType().equals(HiverConstant.DATA_TYPE_ALL)) {
                flagAll = true;
                break;
            }
        }
        // 包含全部权限返回null
        if (flagAll) {
            return null;
        }
        // 每个角色判断 求并集
        for (Role r : roles) {
            if (r.getDataType().equals(HiverConstant.DATA_TYPE_UNDER)) {
                // 本部门及以下
                if (u.getDepartmentId() == 0) {
                    // 用户无部门
                    deparmentIds.add(-1L);
                } else {
                    // 递归获取自己与子级
                    List<Long> ids = new ArrayList<>();
                    getDepRecursion(u.getDepartmentId(), ids);
                    deparmentIds.addAll(ids);
                }
            } else if (r.getDataType().equals(HiverConstant.DATA_TYPE_SAME)) {
                // 本部门
                if (u.getDepartmentId() == 0) {
                    // 用户无部门
                    deparmentIds.add(-1L);
                } else {
                    deparmentIds.add(u.getDepartmentId());
                }
            } else if (r.getDataType().equals(HiverConstant.DATA_TYPE_CUSTOM)) {
                // 自定义
                List<Long> depIds = iUserRoleService.findDepIdsByUserId(u.getId());
                if (depIds == null || depIds.size() == 0) {
                    deparmentIds.add(-1L);
                } else {
                    deparmentIds.addAll(depIds);
                }
            } else if (r.getDataType().equals(HiverConstant.DATA_TYPE_SELF)) {
                // 自己
                deparmentIds.add(-2L);
            }
        }
        // 去重
        LinkedHashSet<Long> set = new LinkedHashSet<>(deparmentIds.size());
        set.addAll(deparmentIds);
        deparmentIds.clear();
        deparmentIds.addAll(set);
        // 缓存
        redisTemplate.set(key, new Gson().toJson(deparmentIds), 15L, TimeUnit.DAYS);
        return deparmentIds;
    }

    /**
     * 获取当前登录用户部分基本信息 id、username、nickname、mobile、email、departmentId、type、permissions（角色和菜单名）
     *
     * @return 当前后台用户
     */
    public User getCurrUserSimple() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null || !authentication.isAuthenticated() || authentication.getName() == null
                || authentication instanceof AnonymousAuthenticationToken) {
            throw new HiverException("未检测到登录用户");
        }
        TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
        User user = new User().setAppId(tokenUser.getAppId()).setUsername(tokenUser.getUsername()).setNickname(tokenUser.getNickname()).
                setMobile(tokenUser.getMobile()).setEmail(tokenUser.getEmail()).setDepartmentId(tokenUser.getDepartmentId()).setType(tokenUser.getType());
        if (tokenUser.getPermissions() != null && !tokenUser.getPermissions().isEmpty()) {
            user.setPermissions(tokenUser.getPermissions().stream().map(e -> new PermissionDTO().setTitle(e)).collect(Collectors.toList()));
        }
        user.setId(tokenUser.getId());
        return user;
    }

    private void getDepRecursion(Long departmentId, List<Long> ids) {
        Department department = departmentDao.getById(departmentId);
        ids.add(department.getId());
        if (department.getIsParent() != null && department.getIsParent()) {
            // 获取其下级
            List<Department> departments =
                    departmentDao.findByParentIdAndStatusOrderBySortOrder(departmentId, HiverConstant.STATUS_NORMAL);
            departments.forEach(d -> {
                getDepRecursion(d.getId(), ids);
            });
        }
    }
}
