/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.aop;

import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.NamedThreadLocal;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.StopWatch;
import cn.hutool.http.useragent.UserAgent;
import cn.hutool.http.useragent.UserAgentUtil;
import plus.hiver.common.annotation.SystemLog;
import plus.hiver.common.entity.Log;
import plus.hiver.common.service.LogService;
import plus.hiver.common.utils.IpInfoUtil;
import plus.hiver.common.utils.ObjectUtil;
import plus.hiver.common.utils.ThreadPoolUtil;
import plus.hiver.common.vo.TokenUser;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@Slf4j
@Aspect
@Component
public class SystemLogAspect {
    private static final ThreadLocal<StopWatch> THREAD_LOCAL_TIMER = new NamedThreadLocal<>("ThreadLocalStopWatch");

    @Autowired
    private LogService logService;

    @Autowired(required = false)
    private HttpServletRequest request;

    @Autowired
    private IpInfoUtil ipInfoUtil;

    /**
     * Controller层切点,注解方式
     */
    @Pointcut("@annotation(plus.hiver.common.annotation.SystemLog)")
    public void controllerAspect() {
    }

    /**
     * 前置通知 (在方法执行之前返回)用于拦截Controller层记录用户的操作的开始时间
     *
     * @param joinPoint 切点
     */
    @Before("controllerAspect()")
    public void doBefore(JoinPoint joinPoint) {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        THREAD_LOCAL_TIMER.set(stopWatch);
    }

    /**
     * 后置通知(在方法执行之后并返回数据) 用于拦截Controller层无异常的操作
     *
     * @param joinPoint 切点
     */
    @AfterReturning("controllerAspect()")
    public void after(JoinPoint joinPoint) {
        try {
            String username = "", device = "", isMobile = "PC端";
            String description = getControllerMethodInfo(joinPoint).get("description").toString();
            int type = (int) getControllerMethodInfo(joinPoint).get("type");
            Map<String, String[]> logParams = request.getParameterMap();
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();

            if (authentication == null || !authentication.isAuthenticated() || authentication.getName() == null
                    || (authentication instanceof AnonymousAuthenticationToken)) {
                return;
            }

            TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
            if(tokenUser != null) {
                username = tokenUser.getUsername();
            } else {
                username = "";
            }

            UserAgent ua = UserAgentUtil.parse(request.getHeader("user-agent"));
            if (ua != null) {
                if (ua.isMobile()) {
                    isMobile = "移动端";
                }
                device = ua.getBrowser().toString() + " " + ua.getVersion() + " | " + ua.getPlatform().toString()
                        + " " + ua.getOs().toString() + " | " + isMobile;
            }

            Log logger = new Log();
            // 请求用户
            logger.setUsername(username);
            // 日志标题
            logger.setName(description);
            // 日志类型
            logger.setLogType(type);
            // 日志请求url
            logger.setRequestUrl(request.getRequestURI());
            // 请求方式
            logger.setRequestType(request.getMethod());
            // 请求参数
            logger.setMapToParams(logParams);
            ipInfoUtil.getInfo(request, ObjectUtil.mapToStringAll(request.getParameterMap()));
            // 请求IP
            logger.setIp(ipInfoUtil.getIpAddr(request));
            // IP地址
            logger.setIpInfo(ipInfoUtil.getIpCity(request));
            // 设备信息
            logger.setDevice(device);

            // 获取计时器并计算耗时
            StopWatch stopWatch = THREAD_LOCAL_TIMER.get();
            if (stopWatch != null && stopWatch.isRunning()) {
                stopWatch.stop();
                logger.setCostTime((int) stopWatch.getTotalTimeMillis());
            }

            // 调用线程保存至ES
            ThreadPoolUtil.getPool().execute(new SaveSystemLogThread(logger, logService));
            THREAD_LOCAL_TIMER.remove();
        } catch (Exception e) {
            log.error("AOP后置通知异常", e);
        }
    }

    /**
     * 保存日志至数据库
     */
    private static class SaveSystemLogThread implements Runnable {
        private final Log logger;
        private final LogService logService;

        public SaveSystemLogThread(Log logger, LogService logService) {
            this.logger = logger;
            this.logService = logService;
        }

        @Override
        public void run() {
            try {
                logService.save(logger);
            } catch (Exception e) {
                log.error("保存日志异常", e);
            }
        }
    }

    /**
     * 获取注解中对方法的描述信息 用于Controller层注解
     *
     * @param joinPoint 切点
     * @return 方法描述
     */
    public static Map<String, Object> getControllerMethodInfo(JoinPoint joinPoint) throws ClassNotFoundException {
        Map<String, Object> map = new ConcurrentHashMap<>(16);
        //获取目标类名
        String targetName = joinPoint.getTarget().getClass().getName();
        //获取方法名
        String methodName = joinPoint.getSignature().getName();
        //获取相关参数
        Object[] arguments = joinPoint.getArgs();
        //生成类对象
        Class<?> targetClass = Class.forName(targetName);
        //获取该类中的方法
        Method[] methods = targetClass.getMethods();

        String description = "";
        Integer type = null;

        for (Method method : methods) {
            if (!method.getName().equals(methodName)) {
                continue;
            }
            Class<?>[] clazzs = method.getParameterTypes();
            if (clazzs.length != arguments.length) {
                continue;
            }
            SystemLog systemLog = method.getAnnotation(SystemLog.class);
            if (systemLog != null) {
                description = systemLog.description();
                type = systemLog.type().ordinal();
                map.put("description", description);
                map.put("type", type);
                break;
            }
        }
        return map;
    }
}
