/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.jpa;

import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.AuditorAware;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import plus.hiver.common.config.security.SecurityUserDetails;
import plus.hiver.common.vo.TokenUser;

import java.util.Optional;

/**
 * 审计记录创建或修改用户
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Configuration
@Slf4j
public class UserAuditor implements AuditorAware<String> {
    @Override
    public Optional<String> getCurrentAuditor() {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null
                    && authentication.isAuthenticated()
                    && !(authentication instanceof AnonymousAuthenticationToken)) {
                TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
                String username = "";
                if(tokenUser != null) {
                    username = tokenUser.getUsername();
                }
                // 只在创建时包含ID
                boolean isCreateOperation = isCreateOperation();
                if (tokenUser != null && isCreateOperation) {
                    Long userId = tokenUser.getUserId();
                    return Optional.of(String.format("%s|%d", username, userId));
                } else {
                    return Optional.of(username);
                }
            }
            return Optional.empty();
        } catch (Exception e) {
            log.warn("Failed to get current auditor: {}", e.getMessage());
            return Optional.empty();
        }
    }

    private Long extractUserId(Authentication authentication) {
        if (authentication.getPrincipal() instanceof SecurityUserDetails) {
            return ((SecurityUserDetails) authentication.getPrincipal()).getId();
        }
        return 0L; // 或者返回null，取决于你的需求
    }

    private boolean isCreateOperation() {
        // 这里需要判断是否是创建操作
        // 可以通过ThreadLocal或请求上下文来判断
        // 或者使用更复杂的方法检测当前操作类型
        return RequestContextHolder.getRequestAttributes() instanceof ServletRequestAttributes
                && ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes())
                .getRequest().getMethod().equalsIgnoreCase("POST");
    }
}
