/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.dao;

import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.stereotype.Repository;
import plus.hiver.common.base.HiverBaseDao;
import plus.hiver.common.entity.User;

import java.util.List;

/**
 * 用户数据处理层
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Repository
public interface UserDao extends HiverBaseDao<User, Long> {
    /**
     * 通过用户名获取用户
     */
    User findByUsername(String username);

    /**
     * 通过手机获取用户
     */
    User findByMobile(String mobile);

    /**
     * 通过邮件获取用户
     */
    User findByEmail(String email);

    /**
     * 通过部门id获取
     */
    List<User> findByDepartmentId(Long departmentId);

    /**
     * 通过用户名模糊搜索
     */
    @Query("select u from User u where u.username like %?1% or u.nickname like %?1% and u.status = ?2")
    List<User> findByUsernameLikeAndStatus(String key, Integer status);

    /**
     * 更新部门名称
     */
    @Modifying
    @Query("update User u set u.departmentTitle=?2 where u.departmentId=?1")
    void updateDepartmentTitle(Long departmentId, String departmentTitle);

    /**
     * 通过部门获得员工列表
     */
    List<User> findByDeletedAndStatusAndDepartmentId(Integer deleted, Integer status, Long departmentId);

    /**
     * 搜索用户
     *
     * @param username
     * @param email
     * @param mobile
     * @return
     */
    User findByUsernameOrEmailOrMobile(String username, String email, String mobile);
}
