/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.utils;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import jakarta.annotation.Resource;
import jakarta.mail.internet.MimeMessage;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.FileSystemResource;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.context.Context;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.service.SettingService;
import plus.hiver.common.vo.EmailSetting;

import java.io.File;
import java.util.Properties;

/**
 * 邮件发送工具类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class EmailUtil {
    @Resource
    private TemplateEngine templateEngine;

    @Resource
    private SettingService settingService;

    /**
     * 构造函数
     *
     * @return
     */
    public EmailSetting getEmailSetting() {
        Setting setting = settingService.findByKey(SettingConstant.EMAIL_SETTING);
        if (StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("您还未配置邮件发送相关配置");
        }
        return new Gson().fromJson(setting.getValue(), EmailSetting.class);
    }

    /**
     * 发送普通邮件
     *
     * @param to
     * @param subject
     * @param htmlContent
     * @param isHtml
     * @param filePaths
     */
    @Async
    public void sendSimpleMail(String to, String subject, String htmlContent, boolean isHtml,
                               String... filePaths) {
        EmailSetting es = getEmailSetting();

        JavaMailSenderImpl senderImpl = new JavaMailSenderImpl();

        // 设定邮箱服务器配置
        senderImpl.setHost(es.getHost());
        senderImpl.setUsername(es.getUsername());
        senderImpl.setPassword(es.getPassword());
        Properties prop = new Properties();

        // 服务器进行认证
        prop.put("mail.smtp.auth", "true");
        prop.put("mail.smtp.timeout", "20000");
        // 邮箱发送服务器端口,这里设置为465端口
        prop.setProperty("mail.smtp.port", "465");
        prop.setProperty("mail.smtp.socketFactory.port", "465");
        prop.put("mail.smtp.ssl.enable", "true");
        senderImpl.setJavaMailProperties(prop);

        // 发送html邮件
        MimeMessage mailMessage = senderImpl.createMimeMessage();
        MimeMessageHelper messageHelper = null;

        // 设置邮件内容
        try {
            messageHelper = new MimeMessageHelper(mailMessage, true, "utf-8");
            messageHelper.setFrom(es.getUsername());
            messageHelper.setTo(to);
            messageHelper.setSubject(subject);
            messageHelper.setText(htmlContent, isHtml);
            // 添加附件
            for (String filePath : filePaths) {
                FileSystemResource file = new FileSystemResource(new File(filePath));
                String fileName = file.getFilename();
                messageHelper.addAttachment(fileName, file);
            }
            senderImpl.send(mailMessage);
            log.info("邮件已发送至：{}", to);
        } catch (Exception e) {
            log.info("邮件：{}失败", to);
            throw new HiverException("发送邮件失败，请检查邮件配置");
        }
    }

    /**
     * 发送带内联资源的邮件（如图片）
     *
     * @param to
     * @param subject
     * @param htmlContent
     * @param resourcePath
     * @param resourceId
     * @param isHtml
     * @param filePaths
     */
    @Async
    public void sendInlineResourceMail(String to, String subject, String htmlContent,
                                       String resourcePath, String resourceId, boolean isHtml, String... filePaths) {
        EmailSetting es = getEmailSetting();

        JavaMailSenderImpl senderImpl = new JavaMailSenderImpl();

        // 设定邮箱服务器配置
        senderImpl.setHost(es.getHost());
        senderImpl.setUsername(es.getUsername());
        senderImpl.setPassword(es.getPassword());
        Properties prop = new Properties();

        // 服务器进行认证
        prop.put("mail.smtp.auth", "true");
        prop.put("mail.smtp.timeout", "20000");
        // 邮箱发送服务器端口,这里设置为465端口
        prop.setProperty("mail.smtp.port", "465");
        prop.setProperty("mail.smtp.socketFactory.port", "465");
        prop.put("mail.smtp.ssl.enable", "true");
        senderImpl.setJavaMailProperties(prop);

        // 发送html邮件
        MimeMessage mailMessage = senderImpl.createMimeMessage();
        MimeMessageHelper messageHelper = null;

        // 设置邮件内容
        try {
            messageHelper = new MimeMessageHelper(mailMessage, true, "utf-8");
            messageHelper.setFrom(es.getUsername());
            messageHelper.setTo(to);
            messageHelper.setSubject(subject);
            messageHelper.setText(htmlContent, isHtml);
            // 添加内联资源
            FileSystemResource res = new FileSystemResource(new File(resourcePath));
            messageHelper.addInline(resourceId, res);
            // 添加附件
            for (String filePath : filePaths) {
                FileSystemResource file = new FileSystemResource(new File(filePath));
                String fileName = file.getFilename();
                messageHelper.addAttachment(fileName, file);
            }
            senderImpl.send(mailMessage);
            log.info("邮件已发送至：{}", to);
        } catch (Exception e) {
            log.info("邮件：{}失败", to);
            throw new HiverException("发送邮件失败，请检查邮件配置");
        }
    }

    /**
     * 使用Thymeleaf模板发送邮件
     *
     * @param to
     * @param subject
     * @param templateName
     * @param context
     * @param filePaths
     */
    @Async
    public void sendTemplateMail(String to, String subject, String templateName, Context context, String... filePaths) {
        EmailSetting es = getEmailSetting();

        JavaMailSenderImpl senderImpl = new JavaMailSenderImpl();

        // 设定邮箱服务器配置
        senderImpl.setHost(es.getHost());
        senderImpl.setUsername(es.getUsername());
        senderImpl.setPassword(es.getPassword());
        Properties prop = new Properties();

        // 服务器进行认证
        prop.put("mail.smtp.auth", "true");
        prop.put("mail.smtp.timeout", "20000");
        // 邮箱发送服务器端口,这里设置为465端口
        prop.setProperty("mail.smtp.port", "465");
        prop.setProperty("mail.smtp.socketFactory.port", "465");
        prop.put("mail.smtp.ssl.enable", "true");
        senderImpl.setJavaMailProperties(prop);

        // 发送html邮件
        MimeMessage mailMessage = senderImpl.createMimeMessage();
        MimeMessageHelper messageHelper = null;

        // 设置邮件内容
        try {
            messageHelper = new MimeMessageHelper(mailMessage, true, "utf-8");
            messageHelper.setFrom(es.getUsername());
            messageHelper.setTo(to);
            messageHelper.setSubject(subject);
            // 添加附件
            for (String filePath : filePaths) {
                FileSystemResource file = new FileSystemResource(new File(filePath));
                String fileName = file.getFilename();
                messageHelper.addAttachment(fileName, file);
            }
            // 处理模板，生成HTML内容
            String emailContent = templateEngine.process(templateName, context);
            // true表示HTML格式的邮件
            messageHelper.setText(emailContent, true);
            // 发送邮件
            senderImpl.send(mailMessage);
            log.info("邮件已发送至：{}", to);
        } catch (Exception e) {
            log.info("邮件：{}失败", to);
            throw new HiverException("发送邮件失败，请检查邮件配置");
        }
    }
}
