/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.vo;

import cn.hutool.core.util.StrUtil;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.dto.RoleDTO;
import plus.hiver.common.entity.User;

import java.util.ArrayList;
import java.util.List;

/**
 * 用户令牌（后台）
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Data
@Builder
@AllArgsConstructor
@NoArgsConstructor
public class TokenUser {
    @Schema(name = "token", title = "令牌")
    private String token;

    @Schema(name = "appId", title = "应用id")
    private String appId;

    @Schema(name = "userId", title = "用户id")
    private Long userId;

    @Schema(name = "username", title = "用户名")
    private String username;

    @Schema(name = "nickname", title = "昵称")
    private String nickname;

    @Schema(name = "mobile", title = "手机号")
    private String mobile;

    @Schema(name = "email", title = "邮箱")
    private String email;

    @Schema(name = "departmentId", title = "所属部门ID")
    private Long departmentId;

    @Schema(name = "type", title = "用户类型 0普通用户 1管理员")
    private Integer type;

    @Schema(name = "permissions", title = "用户拥有的权限")
    private List<PermissionDTO> permissions;

    @Schema(name = "user", title = "登录成功返回内容")
    private User user;

    @Schema(name = "saveLogin", title = "是否保存登录")
    private Boolean saveLogin;

    public TokenUser(User user, Boolean storePerms, Boolean saveLogin) {
        this.appId = user.getAppId();
        this.userId = user.getId();
        this.username = user.getUsername();
        this.nickname = user.getNickname();
        this.mobile = user.getMobile();
        this.email = user.getEmail();
        this.departmentId = user.getDepartmentId();
        this.type = user.getType();
        this.saveLogin = saveLogin;

        // 缓存权限
        List<String> list = new ArrayList<>();
        List<PermissionDTO> permissionDTOList = new ArrayList<>();
        if (storePerms) {
            for (PermissionDTO p : user.getPermissions()) {
                if (StrUtil.isNotBlank(p.getTitle()) && StrUtil.isNotBlank(p.getPath())) {
                    list.add(p.getTitle());
                    permissionDTOList.add(p);
                }
            }
            for (RoleDTO r : user.getRoles()) {
                list.add(r.getName());
            }
            this.setPermissions(permissionDTOList);
        }
    }
}
