/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.dfa.WordTree;
import jakarta.annotation.Nullable;
import jakarta.persistence.criteria.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.dao.StopWordDao;
import plus.hiver.common.entity.StopWord;
import plus.hiver.common.service.StopWordService;
import plus.hiver.common.vo.SearchVo;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 禁用词管理接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class StopWordServiceImpl extends HiverBaseServiceImpl<StopWord, Long, StopWordDao> implements StopWordService {
    public StopWordServiceImpl(StopWordDao dao) {
        super(dao);
    }

    private static WordTree wordTree;

    public WordTree getInstance() {
        if (wordTree == null) {
            // 初始加载数据
            wordTree = new WordTree();
            dao.findAll().forEach(e -> wordTree.addWord(e.getTitle()));
        }
        return wordTree;
    }

    @Override
    public Page<StopWord> findByCondition(StopWord stopWord, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<StopWord>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<StopWord> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> titleField = root.get("title");
                Path<Date> createTimeField = root.get("createTime");

                List<Predicate> list = new ArrayList<>();
                if (StrUtil.isNotBlank(stopWord.getTitle())) {
                    list.add(cb.like(titleField, "%" + stopWord.getTitle() + "%"));
                }
                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }
                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
