/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.utils;

import org.bouncycastle.asn1.gm.GMNamedCurves;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.encoders.Hex;
import plus.hiver.common.constant.HiverConstant;

import java.math.BigInteger;
import java.security.*;
import java.util.Base64;

/**
 * SM解密/解密工具类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
public class SM2Util {
    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    // 获取SM2椭圆曲线参数
    private static X9ECParameters parameters = GMNamedCurves.getByName(HiverConstant.BOUNCY_NAME);
    private static ECDomainParameters domainParameters = new ECDomainParameters(
            parameters.getCurve(), parameters.getG(), parameters.getN(), parameters.getH());
    private static ECParameterSpec ecParameterSpec = new ECParameterSpec(
            parameters.getCurve(), parameters.getG(), parameters.getN(), parameters.getH());

    /**
     * 生成SM2密钥对
     */
    public static KeyPair generateSm2KeyPair() throws Exception {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("EC", "BC");
        keyPairGenerator.initialize(ecParameterSpec, new SecureRandom());
        return keyPairGenerator.generateKeyPair();
    }

    /**
     * 公钥加密
     */
    public static String encrypt(String publicKeyHex, String data) throws Exception {
        // 将16进制公钥转换为椭圆曲线点
        ECPoint ecPoint = parameters.getCurve().decodePoint(Hex.decode(publicKeyHex));
        // 创建公钥参数
        ECPublicKeyParameters publicKeyParameters = new ECPublicKeyParameters(ecPoint, domainParameters);
        // 创建SM2加密引擎
        SM2Engine engine = new SM2Engine();
        engine.init(true, new ParametersWithRandom(publicKeyParameters, new SecureRandom()));
        // 执行加密
        byte[] encrypted = engine.processBlock(data.getBytes(), 0, data.getBytes().length);
        // 返回Base64编码的加密结果
        return Base64.getEncoder().encodeToString(encrypted);
    }

    /**
     * 私钥解密
     */
    public static String decrypt(String privateKeyHex, String encryptedData) throws Exception {
        // 将16进制私钥转换为BigInteger
        BigInteger privateKeyD = new BigInteger(privateKeyHex, 16);
        // 创建私钥参数
        ECPrivateKeyParameters privateKeyParameters = new ECPrivateKeyParameters(privateKeyD, domainParameters);
        // 创建SM2解密引擎
        SM2Engine engine = new SM2Engine();
        engine.init(false, privateKeyParameters);
        // 执行解密
        byte[] decrypted = engine.processBlock(Base64.getDecoder().decode(encryptedData), 0,
                Base64.getDecoder().decode(encryptedData).length);
        // 返回解密结果
        return new String(decrypted);
    }

    /**
     * 从KeyPair中获取16进制公钥
     */
    public static String getPublicKeyHex(KeyPair keyPair) {
        BCECPublicKey publicKey = (BCECPublicKey) keyPair.getPublic();
        return Hex.toHexString(publicKey.getQ().getEncoded(false));
    }

    /**
     * 从KeyPair中获取16进制私钥
     */
    public static String getPrivateKeyHex(KeyPair keyPair) {
        BCECPrivateKey privateKey = (BCECPrivateKey) keyPair.getPrivate();
        return privateKey.getD().toString(16);
    }
}
