/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.vo;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.servlet.http.HttpServletRequest;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.experimental.Accessors;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.CommonUtil;
import plus.hiver.common.utils.IpInfoUtil;

import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 在线配置
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Data
@Accessors(chain = true)
@Schema(description = "在线登录用户信息")
@AllArgsConstructor
public class OnlineUserVo {
    @Schema(description = "令牌")
    private String accessToken;

    @Schema(description = "账号")
    private String username;

    @Schema(description = "ip")
    private String ip;

    @Schema(description = "ip信息")
    private String ipInfo;

    @Schema(description = "设备信息")
    private String device;

    @JsonFormat(timezone = "GMT+8", pattern = "yyyy-MM-dd HH:mm:ss")
    @Schema(description = "第一次活跃时间")
    private Date firstActiveTime;

    @JsonFormat(timezone = "GMT+8", pattern = "yyyy-MM-dd HH:mm:ss")
    @Schema(description = "上次活跃时间")
    private Date lastActiveTime;

    @Schema(description = "持续活跃时间（分钟）")
    private Long activeTime;

    @Schema(description = "是否JWT交互令牌")
    private Boolean isJWT;

    public OnlineUserVo() {
    }

    /**
     * 构造新在线用户
     *
     * @param accessToken
     * @param username
     * @param isJWT
     * @param request
     */
    public OnlineUserVo(String accessToken, String username, Boolean isJWT, HttpServletRequest request) {
        this.accessToken = accessToken;
        this.username = username;
        this.ip = IpInfoUtil.getIpAddr(request);
        this.device = CommonUtil.getDeviceInfo(request);
        Date date = new Date();
        this.firstActiveTime = date;
        this.lastActiveTime = date;
        this.activeTime = 0L;
        this.isJWT = isJWT;
    }

    /**
     * 更新在线用户信息
     *
     * @param accessToken
     * @param username
     * @param isJWT
     * @param redisTemplateHelper
     * @param request
     */
    public static void update(String accessToken, String username, Boolean isJWT, RedisTemplateHelper redisTemplateHelper, HttpServletRequest request) {
        OnlineUserVo onlineUser;
        String key = isJWT ? SecurityConstant.ONLINE_USER_PRE + username : SecurityConstant.ONLINE_USER_PRE + username + ":" + accessToken;
        String v = redisTemplateHelper.get(key);
        if (StrUtil.isBlank(v)) {
            onlineUser = new OnlineUserVo(accessToken, username, isJWT, request);
        } else {
            onlineUser = new Gson().fromJson(v, OnlineUserVo.class);
            // 更新信息
            onlineUser.setIp(IpInfoUtil.getIpAddr(request));
            onlineUser.setDevice(CommonUtil.getDeviceInfo(request));
            // 更新持续活跃时间
            onlineUser.setLastActiveTime(new Date());
            onlineUser.setActiveTime(DateUtil.between(onlineUser.getFirstActiveTime(), onlineUser.getLastActiveTime(), DateUnit.MINUTE));
        }
        redisTemplateHelper.set(key, new Gson().toJson(onlineUser), 30L, TimeUnit.MINUTES);
    }
}
