/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.base;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;

import java.util.List;

/**
 * 数据服务基类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
public interface HiverBaseService<T extends HiverBaseEntity, ID extends Long> {
    /**
     * 根据ID获取 不存在且使用返回的对象时会抛异常
     *
     * @param id 索引
     * @return 实体
     */
    T get(ID id);

    /**
     * 根据ID获取 不存在则返回null
     *
     * @param id 索引
     * @return 实体
     */
    T findById(ID id);

    /**
     * 获取所有列表
     *
     * @return 实体列表
     */
    List<T> getAll();

    /**
     * 获取总数
     *
     * @return 实体总数
     */
    Long getTotalCount();

    /**
     * 保存
     *
     * @param entity 实体
     * @return 实体
     */
    T save(T entity);

    /**
     * 修改
     *
     * @param entity 实体
     * @return 实体
     */
    T update(T entity);

    /**
     * 批量保存与修改
     *
     * @param entities 实体列表
     * @return 实体列表
     */
    List<T> saveOrUpdateAll(List<T> entities);

    /**
     * 删除
     *
     * @param entity 实体
     */
    void delete(T entity);

    /**
     * 根据Id删除
     *
     * @param id 索引
     */
    void delete(ID id);

    /**
     * 批量根据id删除
     *
     * @param ids 索引列表
     */
    void deleteAllById(List<ID> ids);

    /**
     * 批量删除
     *
     * @param entities 实体列表
     */
    void delete(List<T> entities);

    /**
     * 清空缓存，提交持久化
     */
    void flush();

    /**
     * 根据条件查询获取
     *
     * @param spec 查询条件
     * @return 实体列表
     */
    List<T> findAll(Specification<T> spec);

    /**
     * 分页获取
     *
     * @param pageable 分页条件
     * @return 实体列表
     */
    Page<T> findAll(Pageable pageable);

    /**
     * 根据查询条件分页获取
     *
     * @param spec     查询条件
     * @param pageable 分页条件
     * @return 实体列表
     */
    Page<T> findAll(Specification<T> spec, Pageable pageable);

    /**
     * 获取查询条件的结果数
     *
     * @param spec 查询条件
     * @return 结果数
     */
    long count(Specification<T> spec);
}
