/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.sm2.impl;

import jakarta.annotation.Resource;
import org.springframework.stereotype.Service;
import plus.hiver.common.config.sm2.Sm2Config;
import plus.hiver.common.config.sm2.Sm2Service;
import plus.hiver.common.utils.SM2Util;

import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

/**
 * SM2实现类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Service
public class Sm2ServiceImpl implements Sm2Service {
    @Resource
    Sm2Config sm2config;

    @Override
    public String encrypt(String data) {
        if (!sm2config.getEnabled()) {
            return data; // 直接返回原数据
        }
        try {
            return SM2Util.encrypt(sm2config.getPublicKey(), data);
        } catch (Exception e) {
            throw new RuntimeException("SM2加密失败", e);
        }
    }

    @Override
    public String decrypt(String encryptedData) {
        if (!sm2config.getEnabled()) {
            return encryptedData; // 直接返回原数据
        }
        try {
            return SM2Util.decrypt(sm2config.getPrivateKey(), encryptedData);
        } catch (Exception e) {
            throw new RuntimeException("SM2解密失败", e);
        }
    }

    @Override
    public String encodeForUrl(String encrypted) {
        try {
            return URLEncoder.encode(encrypted, StandardCharsets.UTF_8)
                    .replace("+", "%20")  // 空格特殊处理
                    .replace("%7E", "~"); // 保留字符还原
        } catch (Exception e) {
            throw new IllegalStateException("UTF-8编码不支持", e);
        }
    }

    @Override
    public String decodeFromUrl(String encoded) {
        try {
            return URLDecoder.decode(encoded, StandardCharsets.UTF_8);
        } catch (Exception e) {
            throw new IllegalStateException("UTF-8解码不支持", e);
        }
    }

    @Override
    public boolean isEnabled() {
        return Boolean.TRUE.equals(sm2config.getEnabled());
    }
}
