/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.jpa.auditor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.AuditorAware;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import plus.hiver.common.vo.TokenUser;

import java.util.Optional;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Configuration
@Slf4j
public class UserIdAuditor implements AuditorAware<Long> {
    @Override
    public Optional<Long> getCurrentAuditor() {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null
                    && authentication.isAuthenticated()
                    && !(authentication instanceof AnonymousAuthenticationToken)) {
                TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
                Long userId = null;
                if(tokenUser != null) {
                    userId = tokenUser.getUserId();
                    // 只在创建时包含ID
                    boolean isCreateOperation = isCreateOperation();
                    if (isCreateOperation && userId != null) {
                        // 只返回用户名，用于createId字段
                        return Optional.of(userId);
                    }
                }
            }
            return Optional.empty();
        } catch (Exception e) {
            log.warn("Failed to get current user auditor: {}", e.getMessage());
            return Optional.empty();
        }
    }

    private boolean isCreateOperation() {
        // 这里需要判断是否是创建操作
        // 可以通过ThreadLocal或请求上下文来判断
        // 或者使用更复杂的方法检测当前操作类型
        return RequestContextHolder.getRequestAttributes() instanceof ServletRequestAttributes
                && ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes())
                .getRequest().getMethod().equalsIgnoreCase("POST");
    }
}
