/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security;

import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.util.matcher.RegexRequestMatcher;
import org.springframework.web.cors.CorsConfiguration;
import plus.hiver.common.config.properties.IgnoredUrlsProperties;
import plus.hiver.common.config.security.handler.SecurityAccessDeniedHandler;
import plus.hiver.common.config.security.handler.SecurityAuthenticationEntryPoint;
import plus.hiver.common.config.security.service.CustomAuthorizationManagerServiceImpl;
import plus.hiver.common.config.security.validate.EmailValidateFilter;
import plus.hiver.common.config.security.validate.ImageValidateFilter;
import plus.hiver.common.config.security.validate.SmsValidateFilter;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Configuration
@EnableWebSecurity
@EnableMethodSecurity(prePostEnabled = true)
@RequiredArgsConstructor
public class WebSecurityConfig {
    private final IgnoredUrlsProperties ignoredUrlsProperties;
    private final CustomAuthorizationManagerServiceImpl customAuthorizationManagerService;

    private final ImageValidateFilter imageValidateFilter;
    private final SmsValidateFilter smsValidateFilter;
    private final EmailValidateFilter emailValidateFilter;

    public static String[] annotations;

    @Bean
    public SecurityFilterChain filterChain(HttpSecurity httpSecurity) throws Exception {
        // 合并配置文件中的忽略URL和注解标记的忽略URL
        annotations = ignoredUrlsProperties.getUrls();

        httpSecurity
                // 前端段分离不需要---禁用明文验证
                .httpBasic(AbstractHttpConfigurer::disable)
                // 前端段分离不需要---禁用默认登录页
                .formLogin(AbstractHttpConfigurer::disable)
                // 前端段分离不需要---禁用退出页
                .logout(AbstractHttpConfigurer::disable)
                // 前端段分离不需要---csrf攻击
                .csrf(AbstractHttpConfigurer::disable)
                // 跨域访问权限，如果需要可以关闭后自己配置跨域访问
                .cors(cors -> cors.configurationSource(request -> new CorsConfiguration().applyPermitDefaultValues()))
                // 前后端分离采用JWT 不需要session
                .sessionManagement(session ->
                        session.sessionCreationPolicy(SessionCreationPolicy.STATELESS)
                )
                // 前后端分离不需要---记住我
                .rememberMe(AbstractHttpConfigurer::disable)
                // 配置授权规则
                .authorizeHttpRequests(authorize -> authorize
                        // 配置文件中指定的URL放行
                        .requestMatchers(annotations).permitAll()
                        // 静态资源放行
                        .requestMatchers(RegexRequestMatcher.regexMatcher(".*\\.(css|js)$")).permitAll()
                        // 其他请求需要认证
                        .anyRequest().access(customAuthorizationManagerService)
                )
                // 图形验证码过滤器
                .addFilterBefore(imageValidateFilter, UsernamePasswordAuthenticationFilter.class)
                // 短信验证码过滤器
                .addFilterBefore(smsValidateFilter, UsernamePasswordAuthenticationFilter.class)
                // email验证码过滤器
                .addFilterBefore(emailValidateFilter, UsernamePasswordAuthenticationFilter.class)
                // 配置异常处理
                .exceptionHandling(exception -> {
                    // 认证异常处理
                    exception.authenticationEntryPoint(new SecurityAuthenticationEntryPoint());
                    // 授权异常处理
                    exception.accessDeniedHandler(new SecurityAccessDeniedHandler());
                });
        ;

        return httpSecurity.build();
    }
}
