/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security.service;

import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.security.authorization.AuthorizationDecision;
import org.springframework.security.authorization.AuthorizationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.access.intercept.RequestAuthorizationContext;
import org.springframework.stereotype.Service;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.CustomAuthenticationException;
import plus.hiver.common.utils.BaseContext;
import plus.hiver.common.vo.TokenUser;

import java.util.function.Supplier;

/**
 * 自定义授权管理器服务实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Service
public class CustomAuthorizationManagerServiceImpl implements AuthorizationManager<RequestAuthorizationContext> {
    @Resource
    private TokenValidationService tokenValidationService;

    @Resource
    private PermissionCheckService permissionCheckService;

    @Override
    public AuthorizationDecision check(Supplier<Authentication> authentication, RequestAuthorizationContext context) {
        HttpServletRequest request = context.getRequest();
        // 验证token
        TokenUser tokenInfo = tokenValidationService.validateToken(request);
        if(tokenInfo == null) {
            return new AuthorizationDecision(false);
        }
        // 验证用户状态
        validateUserStatus(tokenInfo.getUser());
        // 设置用户信息
        BaseContext.setUsername(tokenInfo.getUsername());
        BaseContext.setUserId(tokenInfo.getUserId());
        BaseContext.setUser(tokenInfo.getUser());
        // 校验权限
        Boolean hasPermission = permissionCheckService.hasPermission(request);
        return new AuthorizationDecision(hasPermission);
    }

    private void validateUserStatus(User user) {
        // 登录信息为空
        if (user == null) {
            throw new CustomAuthenticationException("请先登陆");
        }
        // 判断用户是否禁用
        if (user.getStatus() == -1) {
            throw new CustomAuthenticationException("该账户已封禁");
        }
    }
}
