/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security.service;

import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.stereotype.Component;
import plus.hiver.common.config.properties.IgnoredUrlsProperties;
import plus.hiver.common.dao.mapper.PermissionMapper;
import plus.hiver.common.dao.mapper.UserRoleMapper;
import plus.hiver.common.entity.Permission;
import plus.hiver.common.entity.Role;
import plus.hiver.common.utils.BaseContext;

import java.util.List;
import java.util.Objects;

/**
 * 权限校验认证
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Component
public class PermissionCheckService {
    @Resource
    private UserRoleMapper userRoleMapper;

    @Resource
    private PermissionMapper permissionMapper;

    @Resource
    private IgnoredUrlsProperties ignoredUrlsProperties;

    public Boolean hasPermission(HttpServletRequest request) {
        String requestMethod = request.getMethod();

        // 根据用户ID查询角色数据
        Long userId = BaseContext.getUserId();
        String username = BaseContext.getUsername();

        List<Role> roleList = userRoleMapper.findByUserId(userId);
        for (Role role : roleList) {
            if(role.getName().contains("ROLE_")) return true;
        }

        // 判断请求地址是否是登录之后才需要访问的，已经登录了不需要验证的
        String requestURI = request.getRequestURI();
        for (String url : ignoredUrlsProperties.getUrls()) {
            if (requestURI.contains(url)) return true;
        }

        // 根据角色列表查询权限信息
        List<Permission> permissionList = permissionMapper.findByUserId(userId);

        // 判断是否与请求路径匹配
        return permissionList.stream()
                .map(Permission::getPath)
                .filter(Objects::nonNull)
                .anyMatch(requestUrl -> {
                    // 使用AntPath 匹配
                    if ((requestUrl.contains("/*" ) || requestUrl.contains("/**" ))) {
                        return new AntPathRequestMatcher(requestUrl).matches(request);
                    }
                    // 使用正则匹配（不建议使用）
                    else {
                        return requestURI.matches(requestUrl);
                    }
                });
    }
}
