/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.sms;

import cn.hutool.core.util.StrUtil;
import com.aliyuncs.exceptions.ClientException;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import plus.hiver.common.constant.SettingConstant;
import plus.hiver.common.entity.Setting;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.service.SettingService;

/**
 * 短信工具类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class SmsUtil {
    @Resource
    private SettingService settingService;

    @Resource
    private SmsFactory smsFactory;

    public String getSmsUsed() {
        Setting setting = settingService.findByKey(SettingConstant.SMS_USED);
        if (setting == null || StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("您还未配置短信服务");
        }
        String type = setting.getValue();
        return type;
    }

    /**
     * 获得对应完整短信模版Key
     *
     * @param type
     * @return
     */
    public String getTemplate(String type) {
        return getSmsUsed() + "_" + type;
    }

    public String getTemplateCode(String type) {
        Setting setting = settingService.findByKey(getTemplate(type));
        if (StrUtil.isBlank(setting.getValue())) {
            throw new HiverException("系统还未配置短信服务或相应短信模版");
        }
        return setting.getValue();
    }

    /**
     * 发送验证码 模版变量为${code} 无需模版编号
     *
     * @param mobile
     * @param code
     * @param type   短信模版类型 详见SettingConstant
     * @return
     * @throws ClientException
     */
    public void sendCode(String mobile, String code, String type) {
        sendSms(mobile, "{code:" + code + "}", getTemplateCode(type));
    }

    /**
     * 发送短信
     *
     * @param mobile       手机号 多个,逗号分隔 若为11位国内手机号无需加国家区号86
     *                     国际号码需加上区号 [国家或地区码][手机号] 如8109012345678、86为日本、09012345678为手机号
     * @param params       参数 JSON格式，如{"code": "1234"}
     *                     若启用腾讯短信会自动按顺序转换为逗号分隔的数组值如[1234]
     * @param templateCode 短信模板code/id
     */
    public void sendSms(String mobile, String params, String templateCode) {
        smsFactory.getSms().sendSms(mobile, params, templateCode);
    }
}
