/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.aop;

import jakarta.annotation.Resource;
import lombok.RequiredArgsConstructor;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;
import plus.hiver.common.annotation.Decrypt;
import plus.hiver.common.config.sm2.impl.Sm2ServiceImpl;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;

/**
 * SM2安全AOP
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Aspect
@Component
@RequiredArgsConstructor
public class SecurityAspect {
    @Resource
    private Sm2ServiceImpl sm2Service;

    @Around("@within(plus.hiver.common.annotation.NoAuth) || @annotation(plus.hiver.common.annotation.NoAuth)")
    public Object noAuthCheck(ProceedingJoinPoint joinPoint) throws Throwable {
        return joinPoint.proceed();
    }

    @Around("!@within(plus.hiver.common.annotation.NoAuth) && !@annotation(plus.hiver.common.annotation.NoAuth) && " +
            "(execution(* plus.hiver..controller..*.*(..)))")
    public Object authCheck(ProceedingJoinPoint joinPoint) throws Throwable {
        if (!sm2Service.isEnabled()) {
            return joinPoint.proceed(); // 直接跳过解密
        }

        // 解密参数
        Object[] args = joinPoint.getArgs();
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();
        Parameter[] parameters = method.getParameters();

        for (int i = 0; i < parameters.length; i++) {
            if (parameters[i].isAnnotationPresent(Decrypt.class) && args[i] instanceof String) {
                args[i] = sm2Service.decrypt((String) args[i]);
            }
        }

        // 执行方法
        return joinPoint.proceed(args);
    }
}
