/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security.validate;

import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Resource;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.PathMatcher;
import org.springframework.web.filter.OncePerRequestFilter;
import plus.hiver.common.config.properties.CaptchaProperties;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.ResponseUtil;

import java.io.IOException;

/**
 * 图形验证码过滤器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Configuration
public class ImageValidateFilter extends OncePerRequestFilter {
    @Resource
    private CaptchaProperties captchaProperties;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private PathMatcher pathMatcher;

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws ServletException, IOException {
        // 判断URL是否需要验证
        Boolean flag = false;
        String requestUrl = request.getRequestURI();
        for (String url : captchaProperties.getImage()) {
            if (pathMatcher.match(url, requestUrl)) {
                flag = true;
                break;
            }
        }
        if (flag) {
            String captchaId = request.getParameter("captchaId");
            String code = request.getParameter("code");
            if (StrUtil.isBlank(captchaId) || StrUtil.isBlank(code)) {
                ResponseUtil.resultMap(false, 500, "请传入图形验证码所需参数captchaId或code");
                return;
            }
            String redisCode = redisTemplateHelper.get(captchaId);
            if (StrUtil.isBlank(redisCode)) {
                ResponseUtil.resultMap(false, 500, "验证码已过期，请重新获取");
                return;
            }

            if (!redisCode.toLowerCase().equals(code.toLowerCase())) {
                ResponseUtil.resultMap(false, 500, "图形验证码输入错误");
                return;
            }
            // 已验证清除key
            redisTemplateHelper.delete(captchaId);
            // 验证成功 放行
            chain.doFilter(request, response);
            return;
        }
        // 无需验证 放行
        chain.doFilter(request, response);
    }
}
