/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.utils;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import com.google.common.reflect.TypeToken;
import com.google.gson.Gson;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Component;
import plus.hiver.common.config.properties.HiverTokenProperties;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.constant.UserConstant;
import plus.hiver.common.dao.DepartmentDao;
import plus.hiver.common.dao.UserDao;
import plus.hiver.common.dao.mapper.PermissionMapper;
import plus.hiver.common.dao.mapper.UserRoleMapper;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.dto.RoleDTO;
import plus.hiver.common.entity.Department;
import plus.hiver.common.entity.Permission;
import plus.hiver.common.entity.Role;
import plus.hiver.common.entity.User;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.mybatis.IUserRoleService;
import plus.hiver.common.vo.OnlineUserVo;
import plus.hiver.common.vo.TokenUser;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 安全工具类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Component
public class SecurityUtil {
    @Resource
    private HiverTokenProperties tokenProperties;

    @Resource
    private UserDao userDao;

    @Resource
    private IUserRoleService iUserRoleService;

    @Resource
    private UserRoleMapper userRoleMapper;

    @Resource
    private PermissionMapper permissionMapper;

    @Resource
    private DepartmentDao departmentDao;

    @Resource
    private HttpServletRequest request;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    /**
     * 获取当前用户数据权限 null代表具有所有权限 包含值为-1的数据代表无任何权限 包含值为-2的数据代表仅自己的权限
     */
    public List<Long> getDeparmentIds() {
        List<Long> deparmentIds = new ArrayList<>();
        User u = getCurrUser();
        // 读取缓存
        String key = "userRole::depIds:" + u.getId();
        String v = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(v)) {
            deparmentIds = new Gson().fromJson(v, new TypeToken<List<String>>() {
            }.getType());
            return deparmentIds;
        }
        // 当前用户拥有角色
        List<Role> roles = iUserRoleService.findByUserId(u.getId());
        // 判断有无全部数据的角色
        Boolean flagAll = false;
        for (Role r : roles) {
            if (r.getDataType() == null || r.getDataType().equals(HiverConstant.DATA_TYPE_ALL)) {
                flagAll = true;
                break;
            }
        }
        // 包含全部权限返回null
        if (flagAll) {
            return null;
        }
        // 每个角色判断 求并集
        for (Role r : roles) {
            if (r.getDataType().equals(HiverConstant.DATA_TYPE_UNDER)) {
                // 本部门及以下
                if (u.getDepartmentId() == 0) {
                    // 用户无部门
                    deparmentIds.add(-1L);
                } else {
                    // 递归获取自己与子级
                    List<Long> ids = new ArrayList<>();
                    getDepRecursion(u.getDepartmentId(), ids);
                    deparmentIds.addAll(ids);
                }
            } else if (r.getDataType().equals(HiverConstant.DATA_TYPE_SAME)) {
                // 本部门
                if (u.getDepartmentId() == 0) {
                    // 用户无部门
                    deparmentIds.add(-1L);
                } else {
                    deparmentIds.add(u.getDepartmentId());
                }
            } else if (r.getDataType().equals(HiverConstant.DATA_TYPE_CUSTOM)) {
                // 自定义
                List<Long> depIds = iUserRoleService.findDepIdsByUserId(u.getId());
                if (depIds == null || depIds.size() == 0) {
                    deparmentIds.add(-1L);
                } else {
                    deparmentIds.addAll(depIds);
                }
            } else if (r.getDataType().equals(HiverConstant.DATA_TYPE_SELF)) {
                // 自己
                deparmentIds.add(-2L);
            }
        }
        // 去重
        LinkedHashSet<Long> set = new LinkedHashSet<>(deparmentIds.size());
        set.addAll(deparmentIds);
        deparmentIds.clear();
        deparmentIds.addAll(set);
        // 缓存
        redisTemplateHelper.set(key, new Gson().toJson(deparmentIds), 15L, TimeUnit.DAYS);
        return deparmentIds;
    }

    /**
     * 获取当前登录用户部分基本信息 id、username、nickname、mobile、email、departmentId、type、permissions（角色和菜单名）
     *
     * @return 当前后台用户
     */
    public User getCurrUserSimple() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null || !authentication.isAuthenticated() || authentication.getName() == null
                || authentication instanceof AnonymousAuthenticationToken) {
            throw new HiverException("未检测到登录用户");
        }
        TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
        User user = new User().setUsername(tokenUser.getUsername()).setNickname(tokenUser.getNickname()).
                setMobile(tokenUser.getMobile()).setEmail(tokenUser.getEmail()).setDepartmentId(tokenUser.getDepartmentId()).setType(tokenUser.getType());
        if (tokenUser.getPermissions() != null && !tokenUser.getPermissions().isEmpty()) {
            user.setPermissions(tokenUser.getPermissions().stream().map(e -> new PermissionDTO().setTitle(e.getTitle())).collect(Collectors.toList()));
        }
        user.setId(tokenUser.getUserId());
        return user;
    }

    /**
     * 通过用户名获取用户拥有权限
     *
     * @param username 用户名
     * @return 权限集合
     */
    public List<GrantedAuthority> getCurrUserPerms(String username) {
        List<GrantedAuthority> authorities = new ArrayList<>();
        User user = findUserByUsername(username);
        if (user == null || user.getPermissions() == null || user.getPermissions().isEmpty()) {
            return authorities;
        }
        for (PermissionDTO p : user.getPermissions()) {
            authorities.add(new SimpleGrantedAuthority(p.getTitle()));
        }
        return authorities;
    }

    /**
     * 获取当前登录用户 包含所有信息
     *
     * @return 当前后台用户
     */
    public User getCurrUser() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        if (authentication == null || !authentication.isAuthenticated() || authentication.getName() == null
                || authentication instanceof AnonymousAuthenticationToken) {
            throw new HiverException("未检测到登录用户");
        }
        TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
        return tokenUser.getUser();
    }

    public User findUserByUsername(String username) {
        String key = "username::" + username;
        // 读取缓存
        String res = redisTemplateHelper.get(key);
        if (StrUtil.isNotBlank(res)) {
            return userToDTO(new Gson().fromJson(res, User.class));
        }
        User user = userToDTO(userDao.findByUsername(username));
        // 缓存
        redisTemplateHelper.set(key, new Gson().toJson(user), 15L, TimeUnit.DAYS);
        return user;
    }

    public User userToDTO(User user) {
        if (user == null) {
            return null;
        }
        // 关联角色
        List<Role> roleList = userRoleMapper.findByUserId(user.getId());
        List<RoleDTO> roleDTOList = roleList.stream().map(e ->
                new RoleDTO().setId(e.getId()).setName(e.getName())).collect(Collectors.toList());
        user.setRoles(roleDTOList);
        // 关联权限菜单
        List<Permission> permissionList = permissionMapper.findByUserId(user.getId());
        List<PermissionDTO> permissionDTOList = permissionList.stream()
                .filter(e -> HiverConstant.PERMISSION_OPERATION.equals(e.getType()))
                .map(e -> new PermissionDTO().setTitle(e.getTitle()).setPath(e.getPath())).collect(Collectors.toList());
        user.setPermissions(permissionDTOList);
        return user;
    }

    private void getDepRecursion(Long departmentId, List<Long> ids) {
        Department department = departmentDao.getReferenceById(departmentId);
        ids.add(department.getId());
        if (department.getIsParent() != null && department.getIsParent()) {
            // 获取其下级
            List<Department> departments =
                    departmentDao.findByParentIdAndStatusOrderBySortOrder(departmentId, HiverConstant.STATUS_NORMAL);
            departments.forEach(d -> {
                getDepRecursion(d.getId(), ids);
            });
        }
    }

    public User checkUserPassword(String username, String password) {
        User user;
        // 校验用户名
        if (NameUtil.mobile(username)) {
            user = findUserByMobile(username);
        } else if (NameUtil.email(username)) {
            user = findUserByEmail(username);
        } else {
            user = findUserByUsername(username);
        }
        if (user == null) {
            return null;
        }
        // 校验密码
        Boolean isValid = new BCryptPasswordEncoder().matches(password + user.getSalt(), user.getPassword());
        if (!isValid) {
            return null;
        }
        return user;
    }

    public User findUserByMobile(String mobile) {
        return userToDTO(userDao.findByMobile(mobile));
    }

    public User findUserByEmail(String email) {
        return userToDTO(userDao.findByEmail(email));
    }

    public String getToken(String username, Boolean saveLogin) {
        if (StrUtil.isBlank(username)) {
            throw new HiverException("username不能为空");
        }
        User user = findUserByUsername(username);
        return getToken(user, saveLogin);
    }

    public String getToken(User user, Boolean saveLogin) {
        if (user == null) {
            throw new HiverException("user不能为空");
        }
        if (UserConstant.USER_STATUS_LOCK.equals(user.getStatus())) {
            throw new HiverException("账户被禁用，请联系管理员");
        }
        Boolean saved = false;
        if (saveLogin == null || saveLogin) {
            saved = true;
            if (!tokenProperties.getRedis()) {
                tokenProperties.setTokenExpireTime(tokenProperties.getSaveLoginTime() * 60 * 24);
            }
        }
        // 生成token
        String token;
        TokenUser tokenUser;
        if (tokenProperties.getRedis()) {
            // redis
            token = IdUtil.simpleUUID();
            tokenUser = new TokenUser(user, tokenProperties.getStorePerms(), saved);
            tokenUser.setUser(user);
            tokenUser.setToken(token);
            tokenUser.setSaveLogin(saved);
            // 单设备登录 之前的token失效
            if (tokenProperties.getSdl()) {
                String oldToken = redisTemplateHelper.get(SecurityConstant.USER_TOKEN + user.getUsername());
                if (StrUtil.isNotBlank(oldToken)) {
                    redisTemplateHelper.delete(SecurityConstant.TOKEN_PRE + oldToken);
                }
            }
            // 是否记住账号/保存登录
            if (saved) {
                redisTemplateHelper.set(SecurityConstant.USER_TOKEN + user.getUsername(), token, tokenProperties.getSaveLoginTime(), TimeUnit.DAYS);
                redisTemplateHelper.set(SecurityConstant.TOKEN_PRE + token, new Gson().toJson(tokenUser), tokenProperties.getSaveLoginTime(), TimeUnit.DAYS);
            } else {
                redisTemplateHelper.set(SecurityConstant.USER_TOKEN + user.getUsername(), token, tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
                redisTemplateHelper.set(SecurityConstant.TOKEN_PRE + token, new Gson().toJson(tokenUser), tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
            }
            // 更新在线用户信息
            OnlineUserVo.update(token, user.getUsername(), false, redisTemplateHelper, request);
        } else {
            // JWT 不缓存权限 避免JWT长度过长
            tokenUser = new TokenUser(user, false, null);
            token = JwtTokenUtil.createToken(tokenUser.getUserId(), tokenUser.getUsername(), tokenProperties.getSaveLoginTime());
            tokenUser.setUser(user);
            tokenUser.setToken(token);
            tokenUser.setSaveLogin(saved);
            redisTemplateHelper.set(SecurityConstant.TOKEN_PRE + token, new Gson().toJson(tokenUser), tokenProperties.getSaveLoginTime(), TimeUnit.DAYS);
            OnlineUserVo.update(token, user.getUsername(), true, redisTemplateHelper, request);
        }
        // 记录日志使用
        UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(tokenUser, null, null);
        SecurityContextHolder.getContext().setAuthentication(authentication);
        return token;
    }
}
