/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.aop;

import com.google.gson.Gson;
import jakarta.annotation.Resource;
import lombok.RequiredArgsConstructor;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.stereotype.Component;
import plus.hiver.common.annotation.Encrypt;
import plus.hiver.common.api.Result;
import plus.hiver.common.config.sm2.Sm2Service;
import plus.hiver.common.utils.ResultUtil;

import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

/**
 * SM2加密AOP
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Aspect
@Component
@RequiredArgsConstructor
public class EncryptAspect {
    @Resource
    private Sm2Service sm2Service;

    private final Gson gson = new Gson();

    @Around("@annotation(encrypt)")
    public Object around(ProceedingJoinPoint joinPoint, Encrypt encrypt) throws Throwable {
        if (!sm2Service.isEnabled()) {
            return joinPoint.proceed(); // 直接跳过解密
        }

        // 执行方法
        Object result = joinPoint.proceed();
        // 如果返回结果是Result类型
        if (result instanceof Result<?> originalResult) {
            // 加密数据
            return processGenericResult(originalResult);
        }

        // 其他类型直接加密整个结果
        return sm2Service.encodeForUrl(sm2Service.encrypt(result.toString()));
    }

    private <T> Result<String> processGenericResult(Result<T> result) {
        try {
            // 获取泛型类型
            Type genericType = getGenericType(result);
            // 特殊处理字符串类型
            if (result.getResult() instanceof String stringData) {
                String encrypted = sm2Service.encrypt(stringData);
                return ResultUtil.data(base64Encoder(sm2Service.encodeForUrl(encrypted)));
            }
            // 根据不同类型处理加密
            if (result.getResult() != null) {
                String jsonData = gson.toJson(result.getResult(), genericType);
                String encryptedData = sm2Service.encrypt(jsonData);
                return ResultUtil.data(base64Encoder(sm2Service.encodeForUrl(encryptedData)));
            }
            return ResultUtil.data(null);
        } catch (Exception e) {
            throw new RuntimeException("结果加密失败", e);
        }
    }

    private <T> Type getGenericType(Result<T> result) {
        try {
            // 通过反射获取运行时类型（更通用）
            return result.getClass().getMethod("getResult").getGenericReturnType();
        } catch (NoSuchMethodException e) {
            // 如果无法获取，默认使用Object类型
            return Object.class;
        }
    }

    private String base64Encoder(String data) {
        return Base64.getUrlEncoder().withoutPadding()
                .encodeToString(data.getBytes(StandardCharsets.UTF_8));
    }
}
