/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.persistence.criteria.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.dao.TenantDao;
import plus.hiver.common.entity.Tenant;
import plus.hiver.common.service.TenantService;
import plus.hiver.common.vo.SearchVo;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
public class TenantServiceImpl extends HiverBaseServiceImpl<Tenant, Long, TenantDao> implements TenantService {
    protected TenantServiceImpl(TenantDao dao) {
        super(dao);
    }

    @Override
    public Tenant findByName(String name) {
        return dao.findByName(name);
    }

    @Override
    public List<Tenant> findByStatusAndDeleted(Integer status, Integer deleted) {
        return dao.findByStatusAndDeleted(status, deleted);
    }

    @Override
    public Page<Tenant> findByCondition(Tenant tenant, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<Tenant>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<Tenant> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> nameField = root.get("name");
                Path<String> domainField = root.get("domain");
                Path<String> usernameField = root.get("username");
                Path<String> mobileField = root.get("mobile");
                Path<Integer> statusField = root.get("status");
                Path<Date> createTimeField = root.get("createTime");

                List<Predicate> list = new ArrayList<>();

                // 模糊搜素
                if (StrUtil.isNotBlank(tenant.getName())) {
                    list.add(cb.like(nameField, '%' + tenant.getName() + '%'));
                }
                if (StrUtil.isNotBlank(tenant.getDomain())) {
                    list.add(cb.like(domainField, '%' + tenant.getDomain() + '%'));
                }
                if (StrUtil.isNotBlank(tenant.getUsername())) {
                    list.add(cb.like(usernameField, '%' + tenant.getUsername() + '%'));
                }
                if (StrUtil.isNotBlank(tenant.getMobile())) {
                    list.add(cb.like(mobileField, '%' + tenant.getMobile() + '%'));
                }

                // 状态
                if (tenant.getStatus() != null) {
                    list.add(cb.equal(statusField, tenant.getStatus()));
                }

                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }

                list.add(cb.equal(statusField, 0));
                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }
}
