/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.serviceimpl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Resource;
import jakarta.persistence.criteria.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import plus.hiver.common.base.HiverBaseServiceImpl;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.dao.UserDao;
import plus.hiver.common.dao.mapper.PermissionMapper;
import plus.hiver.common.dao.mapper.UserRoleMapper;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.dto.RoleDTO;
import plus.hiver.common.entity.Permission;
import plus.hiver.common.entity.Role;
import plus.hiver.common.entity.User;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.SearchVo;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户接口实现
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
@Transactional
@CacheConfig(cacheNames = "user")
public class UserServiceImpl extends HiverBaseServiceImpl<User, Long, UserDao> implements UserService {
    private static final String CACHE_NAMES = "user";

    @Resource
    private UserRoleMapper userRoleMapper;

    @Resource
    private PermissionMapper permissionMapper;

    @Resource
    private SecurityUtil securityUtil;

    public UserServiceImpl(UserDao dao) {
        super(dao);
    }

    @Override
    public User findByUsername(String username) {
        User user = dao.findByUsername(username);
        return userToDTO(user);
    }

    @Override
    public User findByMobile(String mobile) {
        User user = dao.findByMobile(mobile);
        return userToDTO(user);
    }

    @Override
    public User findByEmail(String email) {
        User user = dao.findByEmail(email);
        return userToDTO(user);
    }

    public User userToDTO(User user) {
        if (user == null) {
            return null;
        }
        // 关联角色
        List<Role> roleList = userRoleMapper.findByUserId(user.getId());
        List<RoleDTO> roleDTOList = roleList.stream().map(e -> {
            return new RoleDTO().setId(e.getId()).setName(e.getName());
        }).collect(Collectors.toList());
        user.setRoles(roleDTOList);
        // 关联权限菜单
        List<Permission> permissionList = permissionMapper.findByUserId(user.getId());
        List<PermissionDTO> permissionDTOList = permissionList.stream()
                .filter(e -> HiverConstant.PERMISSION_OPERATION.equals(e.getType()))
                .map(e -> {
                    return new PermissionDTO().setTitle(e.getTitle()).setPath(e.getPath());
                }).collect(Collectors.toList());
        user.setPermissions(permissionDTOList);
        return user;
    }

    @Override
    public Page<User> findByCondition(User user, SearchVo searchVo, Pageable pageable) {
        return dao.findAll(new Specification<User>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<User> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> idField = root.get("id");
                Path<String> appIdField = root.get("appId");
                Path<String> usernameField = root.get("username");
                Path<String> nicknameField = root.get("nickname");
                Path<String> mobileField = root.get("mobile");
                Path<String> emailField = root.get("email");
                Path<String> departmentIdField = root.get("departmentId");
                Path<String> sexField = root.get("sex");
                Path<Integer> typeField = root.get("type");
                Path<Integer> statusField = root.get("status");
                Path<Date> createTimeField = root.get("createTime");

                List<Predicate> list = new ArrayList<>();

                /*if (user.getId() != null && user.getId() > 0) {
                    list.add(cb.equal(idField, user.getId()));
                }*/

                if(StrUtil.isNotBlank(user.getAppId())) {
                    list.add(cb.equal(appIdField, user.getAppId()));
                }

                // 模糊搜素
                if (StrUtil.isNotBlank(user.getUsername())) {
                    list.add(cb.like(usernameField, '%' + user.getUsername() + '%'));
                }
                if (StrUtil.isNotBlank(user.getNickname())) {
                    list.add(cb.like(nicknameField, '%' + user.getNickname() + '%'));
                }
                if (StrUtil.isNotBlank(user.getMobile())) {
                    list.add(cb.like(mobileField, '%' + user.getMobile() + '%'));
                }
                if (StrUtil.isNotBlank(user.getEmail())) {
                    list.add(cb.like(emailField, '%' + user.getEmail() + '%'));
                }

                // 部门
                if (user.getDepartmentId() != null && user.getDepartmentId() > 0) {
                    list.add(cb.equal(departmentIdField, user.getDepartmentId()));
                }

                // 性别
                if (StrUtil.isNotBlank(user.getSex())) {
                    list.add(cb.equal(sexField, user.getSex()));
                }
                // 类型
                if (user.getType() != null) {
                    list.add(cb.equal(typeField, user.getType()));
                }
                // 状态
                if (user.getStatus() != null) {
                    list.add(cb.equal(statusField, user.getStatus()));
                }
                // 创建时间
                if (StrUtil.isNotBlank(searchVo.getStartDate()) && StrUtil.isNotBlank(searchVo.getEndDate())) {
                    Date start = DateUtil.parse(searchVo.getStartDate());
                    Date end = DateUtil.parse(searchVo.getEndDate());
                    list.add(cb.between(createTimeField, start, DateUtil.endOfDay(end)));
                }

                // 数据权限
                List<Long> depIds = securityUtil.getDeparmentIds();
                if (depIds != null && depIds.size() > 0) {
                    list.add(departmentIdField.in(depIds));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        }, pageable);
    }

    @Override
    public List<User> findByDepartmentId(Long departmentId) {
        return dao.findByDepartmentId(departmentId);
    }

    @Override
    public List<User> findByUsernameLikeAndStatus(String username, Integer status) {
        return dao.findByUsernameLikeAndStatus(username, status);
    }

    @Override
    public void updateDepartmentTitle(Long departmentId, String departmentTitle) {
        dao.updateDepartmentTitle(departmentId, departmentTitle);
    }

    @Override
    public List<User> findByDeletedAndStatusAndDepartmentId(Integer delFlag, Integer status, Long departmentId) {
        return dao.findByDeletedAndStatusAndDepartmentId(delFlag, status, departmentId);
    }

    @Override
    public List<User> findByAll(User user) {
        return dao.findAll(new Specification<User>() {
            @Nullable
            @Override
            public Predicate toPredicate(Root<User> root, CriteriaQuery<?> cq, CriteriaBuilder cb) {
                Path<String> idField = root.get("id");
                Path<String> appIdField = root.get("appId");
                Path<String> usernameField = root.get("username");
                Path<String> nicknameField = root.get("nickname");
                Path<String> mobileField = root.get("mobile");
                Path<String> emailField = root.get("email");
                Path<String> departmentIdField = root.get("departmentId");
                Path<String> sexField = root.get("sex");
                Path<Integer> typeField = root.get("type");
                Path<Integer> statusField = root.get("status");
                Path<Date> createTimeField = root.get("createTime");

                List<Predicate> list = new ArrayList<>();

                if (user.getId() > 0) {
                    list.add(cb.equal(idField, user.getId()));
                }

                if(StrUtil.isNotBlank(user.getAppId())) {
                    list.add(cb.equal(appIdField, user.getAppId()));
                }

                // 模糊搜素
                if (StrUtil.isNotBlank(user.getUsername())) {
                    list.add(cb.like(usernameField, '%' + user.getUsername() + '%'));
                }
                if (StrUtil.isNotBlank(user.getNickname())) {
                    list.add(cb.like(nicknameField, '%' + user.getNickname() + '%'));
                }
                if (StrUtil.isNotBlank(user.getMobile())) {
                    list.add(cb.like(mobileField, '%' + user.getMobile() + '%'));
                }
                if (StrUtil.isNotBlank(user.getEmail())) {
                    list.add(cb.like(emailField, '%' + user.getEmail() + '%'));
                }

                // 部门
                if (user.getDepartmentId() > 0) {
                    list.add(cb.equal(departmentIdField, user.getDepartmentId()));
                }

                // 性别
                if (StrUtil.isNotBlank(user.getSex())) {
                    list.add(cb.equal(sexField, user.getSex()));
                }
                // 类型
                if (user.getType() != null) {
                    list.add(cb.equal(typeField, user.getType()));
                }
                // 状态
                if (user.getStatus() != null) {
                    list.add(cb.equal(statusField, user.getStatus()));
                }

                // 数据权限
                List<Long> depIds = securityUtil.getDeparmentIds();
                if (depIds != null && depIds.size() > 0) {
                    list.add(departmentIdField.in(depIds));
                }

                Predicate[] arr = new Predicate[list.size()];
                cq.where(list.toArray(arr));
                return null;
            }
        });
    }

    @Override
    public User findByUsernameOrEmailOrMobile(String username, String email, String mobile) {
        return dao.findByUsernameOrEmailOrMobile(username, email, mobile);
    }
}
