/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.jpa;

import jakarta.annotation.Resource;
import jakarta.persistence.PrePersist;
import jakarta.persistence.PreUpdate;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import plus.hiver.common.config.jpa.auditor.TenantIdAuditor;
import plus.hiver.common.config.jpa.auditor.UserAuditor;
import plus.hiver.common.config.jpa.auditor.UserIdAuditor;
import plus.hiver.common.constant.HiverConstant;

import java.lang.reflect.Field;
import java.util.Date;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Configuration
@Slf4j
public class CustomAuditHandler {
    @Resource
    @Qualifier("userAuditor")
    private UserAuditor userAuditor;

    @Resource
    @Qualifier("userIdAuditor")
    private UserIdAuditor userIdAuditor;

    @Resource
    @Qualifier("tenantIdAuditor")
    private TenantIdAuditor tenantIdAuditor;

    /**
     * 插入前处理 - 填充创建相关字段
     *
     * @param entity
     */
    @PrePersist
    public void prePersist(Object entity) {
        try {
            if (isCreateOperation()) {
                if(!entity.getClass().getSimpleName().equals("User")) {
                    // 填充租户ID
                    fillFieldIfPresent(entity, "tenantId", tenantIdAuditor.getCurrentAuditor().orElse(null));
                }
                // 填充创建人ID
                fillFieldIfPresent(entity, "createId", userIdAuditor.getCurrentAuditor().orElse(null));

                // 填充创建人
                fillFieldIfPresent(entity, "createBy", userAuditor.getCurrentAuditor().orElse(null));

                // 填充创建时间
                fillFieldIfPresent(entity, "createTime", new Date());

                // 填充删除标志位
                fillFieldIfPresent(entity, "deleted", HiverConstant.STATUS_NORMAL);
            }
        } catch (Exception e) {
            log.warn("Failed to fill fields during persist: {}", e.getMessage());
        }
    }

    @PreUpdate
    public void preUpdate(Object entity) {
        try {
            // 只填充更新人
            fillFieldIfPresent(entity, "updateBy", userAuditor.getCurrentAuditor().orElse(null));

            // 填充更新时间
            fillFieldIfPresent(entity, "updateTime", new Date());
        } catch (Exception e) {
            log.warn("Failed to fill fields during update: {}", e.getMessage());
        }
    }

    /**
     * 检查实体是否包含指定字段，如果存在则填充值
     *
     * @param entity 实体对象
     * @param fieldName 字段名
     * @param value 要填充的值
     */
    private void fillFieldIfPresent(Object entity, String fieldName, Object value) {
        if (value == null) {
            return;
        }
        try {
            Field field = getField(entity.getClass(), fieldName);
            if (field != null) {
                field.setAccessible(true);
                field.set(entity, value);
            }
        } catch (IllegalAccessException e) {
            log.debug("Field {} not accessible in entity: {}", fieldName, entity.getClass().getName());
        } catch (Exception e) {
            log.debug("Field {} not present or not writable in entity: {}", fieldName, entity.getClass().getName());
        }
    }

    /**
     * 递归获取字段（包括父类）
     */
    private Field getField(Class<?> clazz, String fieldName) {
        try {
            return clazz.getDeclaredField(fieldName);
        } catch (NoSuchFieldException e) {
            Class<?> superClass = clazz.getSuperclass();
            if (superClass != null && !superClass.equals(Object.class)) {
                return getField(superClass, fieldName);
            }
            return null;
        }
    }

    /**
     * 判断是否为创建操作（基于HTTP方法）
     */
    private boolean isCreateOperation() {
        try {
            return RequestContextHolder.getRequestAttributes() instanceof ServletRequestAttributes
                    && ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes())
                    .getRequest().getMethod().equalsIgnoreCase("POST");
        } catch (Exception e) {
            log.debug("无法确定操作类型，默认非创建操作: {}", e.getMessage());
            return false;
        }
    }
}
