/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security;

import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import plus.hiver.common.constant.UserConstant;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.dto.RoleDTO;
import plus.hiver.common.entity.User;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * 用户安全
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
public class SecurityUserDetails extends User implements UserDetails {
    private static final long serialVersionUID = 1L;

    private List<PermissionDTO> permissions;

    private List<RoleDTO> roles;

    public SecurityUserDetails(User user) {
        if (user != null) {
            // Principal用户信息
            this.setId(user.getId());
            this.setTenantId(user.getTenantId());
            this.setUsername(user.getUsername());
            this.setPassword(user.getPassword());
            this.setStatus(user.getStatus());

            this.permissions = user.getPermissions();
            this.roles = user.getRoles();
        }
    }

    /**
     * 添加用户拥有的权限和角色
     *
     * @return 权限集合
     */
    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        List<GrantedAuthority> authorityList = new ArrayList<>();
        // 添加请求权限
        if (permissions != null && permissions.size() > 0) {
            for (PermissionDTO permission : permissions) {
                if (StrUtil.isNotBlank(permission.getTitle()) && StrUtil.isNotBlank(permission.getPath())) {
                    authorityList.add(new SimpleGrantedAuthority(permission.getTitle()));
                }
            }
        }
        // 添加角色
        if (roles != null && roles.size() > 0) {
            // lambda表达式
            roles.forEach(item -> {
                if (StrUtil.isNotBlank(item.getName())) {
                    authorityList.add(new SimpleGrantedAuthority(item.getName()));
                }
            });
        }
        return authorityList;
    }

    /**
     * 账户是否过期
     *
     * @return true：未过期
     */
    @Override
    public boolean isAccountNonExpired() {
        return true;
    }

    /**
     * 是否禁用
     *
     * @return true：未禁用
     */
    @Override
    public boolean isAccountNonLocked() {
        return !UserConstant.USER_STATUS_LOCK.equals(this.getStatus());
    }

    /**
     * 密码是否过期
     *
     * @return true：未过期
     */
    @Override
    public boolean isCredentialsNonExpired() {
        return true;
    }

    /**
     * 是否启用
     *
     * @return true：启用
     */
    @Override
    public boolean isEnabled() {
        return UserConstant.USER_STATUS_NORMAL.equals(this.getStatus());
    }
}
