/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security.interceptor;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.jsonwebtoken.ExpiredJwtException;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerInterceptor;
import plus.hiver.common.config.properties.HiverTokenProperties;
import plus.hiver.common.config.properties.IgnoredUrlsProperties;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.exception.CustomAuthenticationException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.JwtTokenUtil;
import plus.hiver.common.utils.ResponseUtil;
import plus.hiver.common.vo.OnlineUserVo;
import plus.hiver.common.vo.TokenUser;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 令牌刷新拦截器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Component
public class TokenRefreshInterceptor implements HandlerInterceptor {
    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private IgnoredUrlsProperties ignoredUrlsProperties;

    @Resource
    private HiverTokenProperties tokenProperties;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        TokenUser tokenUser = null;
        List<GrantedAuthority> authorities = new ArrayList<>();

        // 排除登录接口和其他不需要拦截的接口
        // 除配置文件忽略路径其它所有请求都需经过认证和授权
        for (String url : ignoredUrlsProperties.getUrls()) {
            if (request.getRequestURI().contains(url)) {
                return true;
            }
        }

        // 判断是否有 token
        String header = request.getHeader(SecurityConstant.HEADER);
        if (header != null && !header.startsWith(SecurityConstant.TOKEN_SPLIT)) {
            ResponseUtil.resultMap(false, 401, "登录已失效，请重新登录");
            return false;
        }

        if (tokenProperties.getRedis()) {
            // Redis
            header = header.replace(SecurityConstant.TOKEN_SPLIT, "");
            String v = redisTemplateHelper.get(SecurityConstant.TOKEN_PRE + header);
            if (StrUtil.isBlank(v)) {
                ResponseUtil.resultMap(false, 401, "会员登录已失效，请重新登录");
                return false;
            }
            tokenUser = new Gson().fromJson(v, TokenUser.class);
            if (!tokenUser.getSaveLogin()) {
                // 更新缓存
                redisTemplateHelper.expire(SecurityConstant.USER_TOKEN + tokenUser.getUsername(), tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
                redisTemplateHelper.expire(SecurityConstant.TOKEN_PRE + v, tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
                redisTemplateHelper.expire(SecurityConstant.TENANT_TOKEN + tokenUser.getUsername(), tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
            }
            // 更新在线用户信息
            OnlineUserVo.updateExpireTime(header, tokenUser.getUsername(), false, redisTemplateHelper, request, tokenProperties.getTokenExpireTime());
            return true;
        } else {
            // JWT
            try {
                String token = header.substring(7);
                // 判断 token 是否过期
                if (JwtTokenUtil.isExpired(token)) {
                    throw new CustomAuthenticationException("身份验证过期");
                }
                // 查找 Redis
                String obj = redisTemplateHelper.get(SecurityConstant.TOKEN_PRE + token);
                tokenUser = new Gson().fromJson(obj, TokenUser.class);
                // 更新在线用户信息
                OnlineUserVo.update(header, tokenUser.getUsername(), true, redisTemplateHelper, request);
                return true;
            } catch (ExpiredJwtException e) {
                ResponseUtil.resultMap(false, 401, "登录已失效，请重新登录");
                return false;
            } catch (Exception e) {
                ResponseUtil.resultMap(false, 500, "解析token错误");
                return false;
            }
        }
    }
}
