/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security.service;

import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.jsonwebtoken.ExpiredJwtException;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;
import plus.hiver.common.config.properties.HiverTokenProperties;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.exception.CustomAuthenticationException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.JwtTokenUtil;
import plus.hiver.common.utils.ResponseUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.common.vo.OnlineUserVo;
import plus.hiver.common.vo.TokenUser;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 处理Token相关逻辑
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Service
public class TokenValidationService {
    @Resource
    private HiverTokenProperties tokenProperties;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private SecurityUtil securityUtil;

    public TokenUser validateToken(HttpServletRequest request) {
        TokenUser tokenUser = null;
        List<GrantedAuthority> authorities = new ArrayList<>();

        // 判断是否有 token
        String header = request.getHeader(SecurityConstant.HEADER);
        if (header != null && !header.startsWith(SecurityConstant.TOKEN_SPLIT)) {
            throw new CustomAuthenticationException("请先登陆");
        }

        if (tokenProperties.getRedis()) {
            // Redis
            header = header.replace(SecurityConstant.TOKEN_SPLIT, "");
            String v = redisTemplateHelper.get(SecurityConstant.TOKEN_PRE + header);
            if (StrUtil.isBlank(v)) {
                ResponseUtil.resultMap(false, 401, "会员登录已失效，请重新登录");
                return null;
            }
            tokenUser = new Gson().fromJson(v, TokenUser.class);
            if (tokenProperties.getStorePerms()) {
                // 缓存了权限
                for (PermissionDTO ga : tokenUser.getPermissions()) {
                    authorities.add(new SimpleGrantedAuthority(ga.getPath()));
                }
            } else {
                // 未缓存 读取权限数据
                authorities = securityUtil.getCurrUserPerms(tokenUser.getUsername());
            }
            if (!tokenUser.getSaveLogin()) {
                // 若未保存登录状态重新设置失效时间
                redisTemplateHelper.set(SecurityConstant.USER_TOKEN + tokenUser.getUsername(), header, tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
                redisTemplateHelper.set(SecurityConstant.TOKEN_PRE + header, v, tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
                redisTemplateHelper.set(SecurityConstant.TENANT_TOKEN + tokenUser.getUsername(), tokenUser.getTenantToken(), tokenProperties.getTokenExpireTime(), TimeUnit.MINUTES);
            }
            // 更新在线用户信息
            OnlineUserVo.updateExpireTime(header, tokenUser.getUsername(), false, redisTemplateHelper, request, tokenProperties.getTokenExpireTime());
        } else {
            // JWT
            try {
                String token = header.substring(7);
                // 判断 token 是否过期
                if (JwtTokenUtil.isExpired(token)) {
                    throw new CustomAuthenticationException("身份验证过期");
                }

                // 解析JWT中的用户名
                String username = JwtTokenUtil.getUsername(token);
                Long userId = JwtTokenUtil.getUserId(token);

                // 查找 Redis
                String obj = redisTemplateHelper.get(SecurityConstant.TOKEN_PRE + token);
                tokenUser = new Gson().fromJson(obj, TokenUser.class);
                // JWT不缓存权限 读取权限数据 避免JWT长度过长
                authorities = securityUtil.getCurrUserPerms(tokenUser.getUsername());
                // 更新在线用户信息
                OnlineUserVo.update(header, tokenUser.getUsername(), true, redisTemplateHelper, request);
            } catch (ExpiredJwtException e) {
                ResponseUtil.resultMap(false, 401, "登录已失效，请重新登录");
            } catch (Exception e) {
                log.error(e.toString());
                ResponseUtil.resultMap(false, 500, "解析token错误");
            }
        }
        if (tokenUser != null && StrUtil.isNotBlank(tokenUser.getUsername())) {
            UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(tokenUser, null, authorities);
            SecurityContextHolder.getContext().setAuthentication(authentication);
            return tokenUser;
        }
        return null;
    }
}
