/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.security.validate;

import cn.hutool.core.util.StrUtil;
import jakarta.annotation.Resource;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.PathMatcher;
import org.springframework.web.filter.OncePerRequestFilter;
import plus.hiver.common.config.properties.CaptchaProperties;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.utils.ResponseUtil;

import java.io.IOException;

/**
 * 短信验证码过滤器
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Configuration
public class SmsValidateFilter extends OncePerRequestFilter {
    @Resource
    private CaptchaProperties captchaProperties;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @Resource
    private PathMatcher pathMatcher;

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws ServletException, IOException {
        // 判断URL是否需要验证
        Boolean flag = false;
        String requestUrl = request.getRequestURI();
        for (String url : captchaProperties.getSms()) {
            if (pathMatcher.match(url, requestUrl)) {
                flag = true;
                break;
            }
        }
        if (flag) {
            String mobile = request.getParameter("mobile");
            String code = request.getParameter("code");
            if (StrUtil.isBlank(mobile) || StrUtil.isBlank(code)) {
                ResponseUtil.resultMap(false, 500, "请传入短信验证码所需参数mobile或code");
                return;
            }
            String redisCode = redisTemplateHelper.get(HiverConstant.PRE_SMS + mobile);
            if (StrUtil.isBlank(redisCode)) {
                ResponseUtil.resultMap(false, 500, "验证码已过期或不存在，请重新获取");
                return;
            }

            if (!redisCode.toLowerCase().equals(code.toLowerCase())) {
                ResponseUtil.resultMap(false, 500, "短信验证码输入错误");
                return;
            }
            // 已验证清除key
            redisTemplateHelper.delete(HiverConstant.PRE_SMS + mobile);
            // 验证成功 放行
            chain.doFilter(request, response);
            return;
        }
        // 无需验证 放行
        chain.doFilter(request, response);
    }
}
