/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.tenant;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import plus.hiver.common.config.properties.TenantProperties;
import plus.hiver.common.utils.BaseContext;

/**
 * 租户处理器 - 处理租户相关的业务逻辑
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class TenantHandler {
    private final TenantProperties tenantProperties;

    public TenantHandler(TenantProperties tenantProperties) {
        this.tenantProperties = tenantProperties;
    }

    /**
     * 设置当前租户上下文
     */
    public void setCurrentTenant(Long tenantId) {
        if (tenantProperties.isEnabled()) {
            BaseContext.setTenantId(tenantId);
            log.debug("Tenant context set to: {}", tenantId);
        } else {
            log.debug("Tenant feature is disabled, ignoring tenant context setting");
        }
    }

    /**
     * 获取当前租户上下文
     */
    public Long getCurrentTenant() {
        if (tenantProperties.isEnabled()) {
            Long tenantId = BaseContext.getTenantId();
            return tenantId != null ? tenantId : tenantProperties.getDefaultTenantId();
        }
        return tenantProperties.getDefaultTenantId();
    }

    /**
     * 清除当前租户上下文
     */
    public void clearCurrentTenant() {
        if (tenantProperties.isEnabled()) {
            BaseContext.removeUser();
            log.debug("Tenant context cleared");
        }
    }

    /**
     * 检查表是否需要忽略租户过滤
     */
    public boolean shouldIgnoreTable(String tableName) {
        return tenantProperties.getIgnoreTables().contains(tableName);
    }

    /**
     * 检查租户功能是否启用
     */
    public boolean isTenantEnabled() {
        return tenantProperties.isEnabled();
    }
}
