/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.config.tenant;

import lombok.extern.slf4j.Slf4j;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.context.spi.CurrentTenantIdentifierResolver;
import org.springframework.boot.autoconfigure.orm.jpa.HibernatePropertiesCustomizer;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import plus.hiver.common.config.properties.TenantProperties;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.utils.BaseContext;
import plus.hiver.common.vo.TokenUser;

import java.util.Map;
import java.util.Optional;

/**
 * 租户标识解析器 - 用于多租户数据隔离
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
public class TenantIdentifierResolver implements CurrentTenantIdentifierResolver<Long>, HibernatePropertiesCustomizer {
    private final TenantProperties tenantProperties;

    public TenantIdentifierResolver(TenantProperties tenantProperties) {
        this.tenantProperties = tenantProperties;
        log.info("Tenant identifier resolver initialized with enabled: {}", tenantProperties.isEnabled());
    }

    /**
     * 默认租户ID（用于系统表或未指定租户的情况）
     */
    private static final Long DEFAULT_TENANT_ID = HiverConstant.DEFAULT_TENANT_ID;

    /**
     * 系统租户ID（用于系统级操作）
     */
    private static final Long SYSTEM_TENANT_ID = HiverConstant.SYSTEM_TENANT_ID;

    @Override
    public Long resolveCurrentTenantIdentifier() {
        // 如果租户功能未启用，返回默认租户ID
        if (!tenantProperties.isEnabled()) {
            return DEFAULT_TENANT_ID;
        }
        try {
            // 首先尝试从安全上下文中获取租户信息
            Optional<Long> securityTenantId = getTenantIdFromSecurityContext();
            if (securityTenantId.isPresent()) {
                return securityTenantId.get();
            }
            // 其次尝试从线程本地上下文中获取
            Long contextTenantId = BaseContext.getTenantId();
            if (contextTenantId != null) {
                return contextTenantId;
            }
            // 最后使用配置的默认租户ID
            log.debug("No tenant ID found in context, using configured default: {}", tenantProperties.getDefaultTenantId());
            return tenantProperties.getDefaultTenantId();
        } catch (Exception e) {
            log.warn("Failed to resolve tenant identifier: {}", e.getMessage());
            return tenantProperties.getDefaultTenantId();
        }
    }

    @Override
    public boolean validateExistingCurrentSessions() {
        return false;
    }

    @Override
    public void customize(Map<String, Object> hibernateProperties) {
        // 仅在租户功能启用时配置Hibernate多租户解析器
        if (tenantProperties.isEnabled()) {
            hibernateProperties.put(AvailableSettings.MULTI_TENANT_IDENTIFIER_RESOLVER, this);
            log.debug("Hibernate multi-tenant configuration applied");
        } else {
            log.debug("Hibernate multi-tenant configuration skipped (tenant feature disabled)");
        }
    }

    /**
     * 从安全上下文中获取租户ID
     */
    private Optional<Long> getTenantIdFromSecurityContext() {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null
                    && authentication.isAuthenticated()
                    && !(authentication instanceof AnonymousAuthenticationToken)) {
                TokenUser tokenUser = (TokenUser) authentication.getPrincipal();
                Long tenantId = null;
                if(tokenUser != null) {
                    tenantId = tokenUser.getTenantId();
                    if (tenantId != null) {
                        log.debug("Resolved tenant ID from security context: {}", tenantId);
                        return Optional.of(tenantId);
                    }
                }
            }
        } catch (Exception e) {
            log.debug("Failed to get tenant ID from security context: {}", e.getMessage());
        }
        return Optional.empty();
    }

    /**
     * 检查是否为系统租户（用于系统级操作）
     */
    public boolean isSystemTenant() {
        Long currentTenant = resolveCurrentTenantIdentifier();
        return SYSTEM_TENANT_ID.equals(currentTenant);
    }

    /**
     * 检查是否为默认租户
     */
    public boolean isDefaultTenant() {
        Long currentTenant = resolveCurrentTenantIdentifier();
        return DEFAULT_TENANT_ID.equals(currentTenant);
    }

    /**
     * 检查租户功能是否启用
     */
    public boolean isTenantEnabled() {
        return tenantProperties.isEnabled();
    }
}
