/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.entity;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableName;
import com.fasterxml.jackson.annotation.JsonFormat;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.Table;
import jakarta.persistence.Transient;
import jakarta.validation.constraints.NotNull;
import jakarta.validation.constraints.Pattern;
import jakarta.validation.constraints.Size;
import lombok.Data;
import lombok.experimental.Accessors;
import org.hibernate.annotations.DynamicInsert;
import org.hibernate.annotations.DynamicUpdate;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import plus.hiver.common.base.HiverBaseEntity;
import plus.hiver.common.constant.HiverConstant;
import plus.hiver.common.constant.UserConstant;
import plus.hiver.common.dto.PermissionDTO;
import plus.hiver.common.dto.RoleDTO;
import plus.hiver.common.utils.NameUtil;

import java.util.Collection;
import java.util.Date;
import java.util.List;

/**
 * 后台用户实体类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Data
@Entity
@DynamicInsert
@DynamicUpdate
@Accessors(chain = true)
@Table(name = "t_user")
@TableName("t_user")
@Tag(name = "用户")
public class User extends HiverBaseEntity implements UserDetails {
    private static final long serialVersionUID = 1L;

    @Schema(description = "租户ID")
    private Long tenantId;

    @Transient
    @TableField(exist = false)
    @Schema(description = "租户令牌")
    private String tenantToken;

    @Schema(description = "登录名")
    @Column(unique = true, nullable = false)
    @Pattern(regexp = NameUtil.regUsername, message = "登录账号不能包含中文、特殊字符 长度不能>16")
    private String username;

    @Schema(description = "密码")
    @NotNull(message = "不能为空")
    private String password;

    @Schema(description = "用户名/昵称/姓名")
    @Size(max = 20, message = "昵称长度不能超过20")
    private String nickname;

    @Schema(description = "手机")
    @Pattern(regexp = NameUtil.regMobile, message = "11位手机号格式不正确")
    private String mobile;

    @Schema(description = "邮箱")
    @Pattern(regexp = NameUtil.regEmail, message = "邮箱格式不正确")
    private String email;

    @Schema(description = "省市县地址")
    private String address;

    @Schema(description = "街道地址")
    private String street;

    @Schema(description = "性别")
    private String sex;

    @Schema(description = "密码强度")
    @Column(length = 2)
    private String passStrength;

    @Schema(description = "用户头像")
    private String avatar = UserConstant.USER_DEFAULT_AVATAR;

    @Schema(description = "用户类型 0普通用户 1管理员")
    private Integer type;

    @Schema(description = "状态 默认0正常 -1拉黑")
    private Integer status;

    @Schema(description = "描述/详情/备注")
    private String description;

    @Schema(description = "所属部门id")
    private Long departmentId;

    @Schema(description = "所属部门名称")
    private String departmentTitle;

    @JsonFormat(timezone = HiverConstant.GMT8, pattern = "yyyy-MM-dd")
    @DateTimeFormat(pattern = "yyyy-MM-dd")
    @Schema(description = "生日")
    private Date birth;

    @Schema(description = "盐")
    private String salt;

    @Transient
    @TableField(exist = false)
    @Schema(description = "用户拥有角色")
    private List<RoleDTO> roles;

    @Transient
    @TableField(exist = false)
    @Schema(description = "用户拥有的权限")
    private List<PermissionDTO> permissions;

    @Transient
    @TableField(exist = false)
    @Schema(description = "导入数据时使用")
    private Integer defaultRole;

    @Transient
    @TableField(exist = false)
    @Schema(description = "角色ID列表")
    private Long[] roleIds;

    /**
     * 是否拥有某角色
     *
     * @param title 角色名称
     * @return true 有
     */
    public Boolean hasRole(String title) {
        if (StrUtil.isBlank(title)) {
            return false;
        }
        for (RoleDTO r : this.roles) {
            if (title.equals(r.getName())) {
                return true;
            }
        }
        return false;
    }

    @Transient
    @TableField(exist = false)
    @Schema(description = "用户权限列表")
    private List<GrantedAuthority> authorities;

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        return authorities;
    }
}
