/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.common.utils;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.annotation.PostConstruct;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.ClassPathResource;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.*;

/**
 * 行政区划编码工具类
 *
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@Component
public class RegionCodeUtil {
    private static final String REGION_DATA_PATH = "data/regions.json";
    private static final ObjectMapper objectMapper = new ObjectMapper();

    /**
     * 最外层 Map：
     * key: 上级编码（如 "86"、"120000"、"120100"）
     * value: Map<下级编码, 下级名称> （如 {"120100": "市辖区", "120106": "红桥区"}）
     */
    private static Map<String, Map<String, String>> levelMap = new HashMap<>();

    @PostConstruct
    public void init() {
        try {
            // 1. 读取 JSON 文件
            ClassPathResource resource = new ClassPathResource(REGION_DATA_PATH);
            // 2. 解析为：Map<上级编码, Map<下级编码, 下级名称>>
            Map<String, Map<String, String>> regionData = objectMapper.readValue(
                    resource.getInputStream(),
                    new TypeReference<Map<String, Map<String, String>>>() {}
            );
            // 3. 构建层级映射表 levelMap
            levelMap = regionData;
            log.info("行政区划编码工具类初始化完成，共加载 {} 个上级节点", levelMap.size());
        } catch (IOException e) {
            log.error("初始化行政区划数据失败", e);
            throw new RuntimeException("行政区划数据初始化失败", e);
        }
    }

    /**
     * 根据 "省,市,区" 格式的地址字符串，返回对应的编码字符串
     * 如输入 "天津市,市辖区,红桥区"，返回 "120000,120100,120106"
     *
     * @param addressStr 如 "天津市,市辖区,红桥区"
     * @return 编码字符串，如 "120000,120100,120106"；若某部分找不到则保留原名称
     */
    public static String getCodesByAddress(String addressStr) {
        if (addressStr == null || addressStr.trim().isEmpty()) {
            return null;
        }
        // 拆分输入，如 ["天津市", "市辖区", "红桥区"]
        String[] names = addressStr.split("\\s*,\\s*");
        List<String> codeList = new ArrayList<>();
        // 起始上级编码是国家节点 "86"
        String currentParentCode = "86";
        for (String name : names) {
            // 1. 在当前父级下，查找名称对应的编码
            String code = findCodeByName(currentParentCode, name);
            if (code == null) {
                // 找不到就保留原名称，并停止后续查找（因为不知道下一级的父级了）
                codeList.add(name);
                log.warn("未找到名称 '{}' 对应的编码，当前父级编码: {}", name, currentParentCode);
                break;
            } else {
                // 2. 找到了，添加到结果列表
                codeList.add(code);
                // 3. 更新下一级的父级编码，继续查找下一级（如从 "120000" 找 "市辖区"，然后从 "120100" 找 "红桥区"）
                currentParentCode = code;
            }
        }
        // 4. 拼接编码结果，如 "120000,120100,120106"
        return String.join(",", codeList);
    }

    /**
     * 在指定的父级编码下，查找名称对应的编码
     *
     * @param parentCode 上级编码，如 "86"、"120000"、"120100"
     * @param name       要查找的名称，如 "天津市"、"市辖区"、"红桥区"
     * @return 对应的编码，如 "120000"、"120100"、"120106"；找不到返回 null
     */
    private static String findCodeByName(String parentCode, String name) {
        // 1. 获取该父级下的所有子级：Map<编码, 名称>
        Map<String, String> children = levelMap.get(parentCode);
        if (children == null) {
            return null;
        }
        // 2. 遍历所有子级，找到名称匹配的编码
        for (Map.Entry<String, String> entry : children.entrySet()) {
            String code = entry.getKey();
            String value = entry.getValue();
            if (value != null && value.equals(name)) {
                return code; // 找到匹配的编码
            }
        }
        // 3. 未找到
        return null;
    }

    /**
     * 【可选】通过单个名称获取编码（不推荐，因为有重名问题）
     *
     * @param name
     * @return
     */
    public static String getCodeByName(String name) {
        // 注意：有重名风险，不建议直接使用
        for (Map<String, String> children : levelMap.values()) {
            for (Map.Entry<String, String> entry : children.entrySet()) {
                if (entry.getValue() != null && entry.getValue().equals(name)) {
                    return entry.getKey();
                }
            }
        }
        return null;
    }
}
