/*
 * Copyright 2023-2025 Licensed under the AGPL License
 */
package plus.hiver.module.open.controller;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import com.google.gson.Gson;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import plus.hiver.common.annotation.PermissionTag;
import plus.hiver.common.annotation.SystemLog;
import plus.hiver.common.api.Result;
import plus.hiver.common.constant.OAuthConstant;
import plus.hiver.common.constant.SecurityConstant;
import plus.hiver.common.entity.User;
import plus.hiver.common.enums.LogType;
import plus.hiver.common.exception.HiverException;
import plus.hiver.common.redis.RedisTemplateHelper;
import plus.hiver.common.service.UserService;
import plus.hiver.common.utils.ResultUtil;
import plus.hiver.common.utils.SecurityUtil;
import plus.hiver.module.open.entity.Client;
import plus.hiver.module.open.service.ClientService;
import plus.hiver.module.open.vo.Oauth2TokenInfo;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * <p>
 * 尊重知识产权，CV 请保留版权，海文科技 https://hiver.cc 出品，不允许非法使用，后果自负
 * </p>
 *
 * @author Yazhi Li
 */
@Slf4j
@RestController
@AllArgsConstructor
@Tag(name = "OAuth2认证接口")
@PermissionTag(permission = "oauth2:*" )
@RequestMapping("/hiver/oauth2")
public class Oauth2Controller {
    @Resource
    private ClientService clientService;

    @Resource
    private UserService userService;

    @Resource
    private SecurityUtil securityUtil;

    @Resource
    private RedisTemplateHelper redisTemplateHelper;

    @GetMapping(value = "/info/{client_id}")
    @Operation(summary = "站点基本信息")
    public Result info(@Parameter(description ="客户端id") @PathVariable Long client_id) {
        Client client = getClient(client_id);
        Map<String, Object> map = new HashMap<>(16);
        map.put("name", client.getName());
        map.put("homeUri", client.getHomeUri());
        map.put("logo", client.getLogo());
        map.put("autoApprove", client.getAutoApprove());
        return ResultUtil.data(map);
    }

    @PostMapping(value = "/authorize")
    @Operation(summary = "认证获取code")
    @SystemLog(description = "认证中心登录", type = LogType.LOGIN)
    public Result authorize(@Parameter(description ="用户名") @RequestParam String username,
                            @Parameter(description ="密码") @RequestParam String password,
                            @Parameter(description ="客户端id") @RequestParam Long client_id,
                            @Parameter(description ="成功授权后回调地址") @RequestParam String redirect_uri,
                            @Parameter(description ="授权类型为code") @RequestParam(required = false, defaultValue = "code") String response_type,
                            @Parameter(description ="客户端状态值") @RequestParam String state) {
        Client client = getClient(client_id);

        if (!client.getRedirectUri().equals(redirect_uri)) {
            return ResultUtil.error("回调地址redirect_uri不正确");
        }
        // 登录认证
        User user = securityUtil.checkUserPassword(username, password);
        if (user == null) {
            return ResultUtil.error("账号或密码错误");
        }

        String accessToken = securityUtil.getToken(user, true);
        // 生成code 5分钟内有效
        String code = IdUtil.simpleUUID();
        // 存入用户及clientId信息
        redisTemplateHelper.set(OAuthConstant.OAUTH_CODE_PRE + code,
                new Gson().toJson(new Oauth2TokenInfo(client_id, user.getUsername())), 5L, TimeUnit.MINUTES);
        Map<String, Object> map = new HashMap<>(16);
        map.put("code", code);
        map.put("redirect_uri", redirect_uri);
        map.put("state", state);
        map.put("accessToken", accessToken);
        return ResultUtil.data(map);
    }

    @GetMapping(value = "/token")
    @Operation(summary = "获取access_token令牌")
    public Result token(@Parameter(description ="授权类型") @RequestParam String grant_type,
                        @Parameter(description ="客户端id") @RequestParam Long client_id,
                        @Parameter(description ="客户端秘钥") @RequestParam String client_secret,
                        @Parameter(description ="认证返回的code") @RequestParam(required = false) String code,
                        @Parameter(description ="刷新token") @RequestParam(required = false) String refresh_token,
                        @Parameter(description ="成功授权后回调地址") @RequestParam(required = false) String redirect_uri) {
        Client client = getClient(client_id);
        // 判断clientSecret
        if (!client.getClientSecret().equals(client_secret)) {
            return ResultUtil.error("client_secret不正确");
        }
        Oauth2TokenInfo tokenInfo = null;
        if (OAuthConstant.AUTHORIZATION_CODE.equals(grant_type)) {
            // 判断回调地址
            if (!client.getRedirectUri().equals(redirect_uri)) {
                return ResultUtil.error("回调地址redirect_uri不正确");
            }
            // 判断code 获取用户信息
            String codeValue = redisTemplateHelper.get(OAuthConstant.OAUTH_CODE_PRE + code);
            if (StrUtil.isBlank(codeValue)) {
                return ResultUtil.error("code已过期");
            }
            tokenInfo = new Gson().fromJson(codeValue, Oauth2TokenInfo.class);
            if (!tokenInfo.getClientId().equals(client_id)) {
                return ResultUtil.error("code不正确");
            }
        } else if (OAuthConstant.REFRESH_TOKEN.equals(grant_type)) {
            // 从refreshToken中获取用户信息
            String refreshTokenValue = redisTemplateHelper.get(OAuthConstant.OAUTH_TOKEN_INFO_PRE + refresh_token);
            if (StrUtil.isBlank(refreshTokenValue)) {
                return ResultUtil.error("refresh_token已过期");
            }
            tokenInfo = new Gson().fromJson(refreshTokenValue, Oauth2TokenInfo.class);
            if (!tokenInfo.getClientId().equals(client_id)) {
                return ResultUtil.error("refresh_token不正确");
            }
        } else {
            return ResultUtil.error("授权类型grant_type不正确");
        }
        String token = null, refreshToken = null;
        Long expiresIn = null;
        String tokenKey = OAuthConstant.OAUTH_TOKEN_PRE + tokenInfo.getUsername() + ":" + client_id,
                refreshKey = OAuthConstant.OAUTH_REFRESH_TOKEN_PRE + tokenInfo.getUsername() + ":" + client_id;
        if (OAuthConstant.AUTHORIZATION_CODE.equals(grant_type)) {
            // 生成token模式
            String oldToken = redisTemplateHelper.get(tokenKey);
            String oldRefreshToken = redisTemplateHelper.get(refreshKey);
            if (StrUtil.isNotBlank(oldToken) && StrUtil.isNotBlank(oldRefreshToken)) {
                // 旧token
                token = oldToken;
                refreshToken = oldRefreshToken;
                expiresIn = redisTemplateHelper.getExpire(OAuthConstant.OAUTH_TOKEN_INFO_PRE + token, TimeUnit.SECONDS);
            } else {
                // 新生成 30天过期
                String newToken = IdUtil.simpleUUID();
                String newRefreshToken = IdUtil.simpleUUID();
                redisTemplateHelper.set(tokenKey, newToken, 30L, TimeUnit.DAYS);
                redisTemplateHelper.set(refreshKey, newRefreshToken, 30L, TimeUnit.DAYS);
                // 新token中存入用户信息
                redisTemplateHelper.set(OAuthConstant.OAUTH_TOKEN_INFO_PRE + newToken, new Gson().toJson(tokenInfo), 30L, TimeUnit.DAYS);
                redisTemplateHelper.set(OAuthConstant.OAUTH_TOKEN_INFO_PRE + newRefreshToken, new Gson().toJson(tokenInfo), 30L, TimeUnit.DAYS);
                token = newToken;
                refreshToken = newRefreshToken;
                expiresIn = redisTemplateHelper.getExpire(OAuthConstant.OAUTH_TOKEN_INFO_PRE + token, TimeUnit.SECONDS);
            }
        } else if (OAuthConstant.REFRESH_TOKEN.equals(grant_type)) {
            // 刷新token模式 生成新token 30天过期
            String newToken = IdUtil.simpleUUID();
            String newRefreshToken = IdUtil.simpleUUID();
            redisTemplateHelper.set(tokenKey, newToken, 30L, TimeUnit.DAYS);
            redisTemplateHelper.set(refreshKey, newRefreshToken, 30L, TimeUnit.DAYS);
            // 新token中存入用户信息
            redisTemplateHelper.set(OAuthConstant.OAUTH_TOKEN_INFO_PRE + newToken, new Gson().toJson(tokenInfo), 30L, TimeUnit.DAYS);
            redisTemplateHelper.set(OAuthConstant.OAUTH_TOKEN_INFO_PRE + newRefreshToken, new Gson().toJson(tokenInfo), 30L, TimeUnit.DAYS);
            token = newToken;
            refreshToken = newRefreshToken;
            expiresIn = redisTemplateHelper.getExpire(OAuthConstant.OAUTH_TOKEN_INFO_PRE + token, TimeUnit.SECONDS);
            // 旧refreshToken过期
            redisTemplateHelper.delete(OAuthConstant.OAUTH_TOKEN_INFO_PRE + refresh_token);
        }

        Map<String, Object> map = new HashMap<>(16);
        map.put("access_token", token);
        map.put("expires_in", expiresIn);
        map.put("refresh_token", refreshToken);
        return ResultUtil.data(map);
    }

    @PostMapping(value = "/authorized")
    @Operation(summary = "已认证过获取code/单点登录实现")
    public Result authorized(@Parameter(description ="客户端id") @RequestParam Long client_id,
                             @Parameter(description ="成功授权后回调地址") @RequestParam String redirect_uri,
                             @Parameter(description ="客户端状态值") @RequestParam String state) {
        Client client = getClient(client_id);
        // 判断回调地址
        if (!client.getRedirectUri().equals(redirect_uri)) {
            return ResultUtil.error("回调地址redirect_uri不正确");
        }
        User user = securityUtil.getCurrUserSimple();
        // 生成code 5分钟内有效
        String code = IdUtil.simpleUUID();
        redisTemplateHelper.set(OAuthConstant.OAUTH_CODE_PRE + code,
                new Gson().toJson(new Oauth2TokenInfo(client_id, user.getUsername())), 5L, TimeUnit.MINUTES);
        Map<String, Object> map = new HashMap<>(16);
        map.put("code", code);
        map.put("redirect_uri", redirect_uri);
        map.put("state", state);
        return ResultUtil.data(map);
    }

    @PostMapping(value = "/logout")
    @Operation(summary = "退出登录（内部信任站点使用）")
    public Result logout() {
        User user = securityUtil.getCurrUserSimple();
        // 删除当前用户登录accessToken
        String token = redisTemplateHelper.get(SecurityConstant.USER_TOKEN + user.getUsername());
        redisTemplateHelper.delete(token);
        redisTemplateHelper.delete(SecurityConstant.USER_TOKEN + user.getUsername());
        // 删除当前用户授权第三方应用的access_token
        redisTemplateHelper.deleteByPattern(OAuthConstant.OAUTH_TOKEN_PRE + user.getUsername() + ":*");
        redisTemplateHelper.deleteByPattern(OAuthConstant.OAUTH_REFRESH_TOKEN_PRE + user.getUsername() + ":*");
        return ResultUtil.data(null);
    }

    @GetMapping(value = "/user")
    @Operation(summary = "获取用户信息")
    public Result user(@Parameter(description ="令牌") @RequestParam String access_token) {
        String tokenValue = redisTemplateHelper.get(OAuthConstant.OAUTH_TOKEN_INFO_PRE + access_token);
        if (StrUtil.isBlank(tokenValue)) {
            return ResultUtil.error("access_token已过期失效");
        }
        Oauth2TokenInfo tokenInfo = new Gson().fromJson(tokenValue, Oauth2TokenInfo.class);
        User user = userService.findByUsername(tokenInfo.getUsername());
        if (user == null) {
            return ResultUtil.error("用户信息不存在");
        }
        Map<String, Object> map = new HashMap<>(16);
        map.put("username", tokenInfo.getUsername());
        map.put("avatar", user.getAvatar());
        map.put("sex", user.getSex());
        return ResultUtil.data(map);
    }

    private Client getClient(Long client_id) {
        Client client = clientService.findById(client_id);
        if (client == null) {
            throw new HiverException("客户端client_id不存在");
        }
        return client;
    }
}
